<?php

namespace Drupal\optimize_database_tables\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service responsible for low-level database operations for optimization.
 *
 * Provides methods to list database tables and run the SQL OPTIMIZE command
 * on a given table. This service is used by higher-level orchestration code
 * to perform bulk optimization via Drupal batches.
 */
class DbHandler {

  /**
   * The active database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * Logger channel factory for writing errors to Drupal logs.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $logger;

  /**
   * Constructs the database handler service.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger
   *   The logger channel factory.
   */
  public function __construct(
    Connection $database,
    LoggerChannelFactoryInterface $logger,
  ) {
    $this->database = $database;
    $this->logger = $logger;
  }

  /**
   * Returns a list of base tables in the current database.
   *
   * @return array
   *   An associative array of table names keyed by table name.
   */
  public function getTablesList(): array {
    $tables = [];
    try {
      $query = $this->database
        ->query("SHOW FULL TABLES WHERE Table_type = 'BASE TABLE'");
      $result = $query->fetchAll();
      foreach ($result as $table_name) {
        $table = current((array) $table_name);
        $tables[$table] = $table;
      }
    }
    catch (\Exception $e) {
      $this->logger->get('optimize_database_tables')->error($e->getMessage());
    }

    return $tables;
  }

  /**
   * Optimizes a single database table.
   *
   * @param string $table_name
   *   The name of the table to optimize.
   *
   * @return void
   *   No return value. Executes an OPTIMIZE TABLE statement.
   */
  public function optimizeTable(string $table_name): void {
    try {
      $this->database->query("OPTIMIZE TABLE {$table_name}")->execute();
    }
    catch (\Exception $e) {
      $this->logger->get('optimize_database_tables')->error($e->getMessage());
    }
  }

  /**
   * Returns table size in bytes for a given table (data + indexes).
   *
   * @param string $table_name
   *   The table name.
   *
   * @return int
   *   The total size of the table in bytes.
   */
  public function getTableSizeBytes(string $table_name): int {
    try {
      $schema = $this->database->getConnectionOptions()['database'] ?? '';
      $result = $this->database->query(
        'SELECT COALESCE(DATA_LENGTH,0) + COALESCE(INDEX_LENGTH,0) AS size
         FROM INFORMATION_SCHEMA.TABLES
         WHERE TABLE_SCHEMA = :schema AND TABLE_NAME = :table',
        [':schema' => $schema, ':table' => $table_name]
      )->fetchField();
      return (int) $result;
    }
    catch (\Exception $e) {
      $this->logger->get('optimize_database_tables')->error($e->getMessage());
      return 0;
    }
  }

  /**
   * Returns total size (bytes) of a set of tables.
   *
   * @param string[] $tables
   *   The list of table names.
   *
   * @return int
   *   The sum of sizes (bytes) for the provided tables.
   */
  public function getTotalSizeBytes(array $tables): int {
    $total = 0;
    foreach ($tables as $table) {
      $name = is_string($table) ? $table : (string) $table;
      $total += $this->getTableSizeBytes($name);
    }
    return $total;
  }

  /**
   * Formats bytes to a human-readable string (e.g., 12.3 MB).
   *
   * @param int $bytes
   *   The number of bytes.
   * @param int $precision
   *   The number of decimal digits to include.
   *
   * @return string
   *   The formatted size string.
   */
  public function formatBytes(int $bytes, int $precision = 2): string {
    if ($bytes <= 0) {
      return '0 B';
    }
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $power = (int) floor(log($bytes, 1024));
    $power = min($power, count($units) - 1);
    $value = $bytes / (1024 ** $power);
    return number_format($value, $precision) . ' ' . $units[$power];
  }

}
