<?php

namespace Drupal\optimize_database_tables\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\optimize_database_tables\Service\DbHandler;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for the Optimize DB module.
 *
 * Allows administrators to choose whether to optimize all tables or only a
 * selected subset, stored in module configuration.
 */
class OptimizeDatabaseTablesConfigForm extends ConfigFormBase {

  /**
   * Configuration object name used by the module settings.
   */
  public const string OPTIMIZE_DATABASE_SETTINGS = 'optimize_database_tables.settings';

  /**
   * Service for retrieving available tables and optimizing them.
   *
   * @var \Drupal\optimize_database_tables\Service\DbHandler
   */
  protected DbHandler $dbHandler;

  /**
   * Constructs the configuration form for Optimize DB.
   *
   * Injects the configuration factory and typed configuration manager
   * required by the parent ConfigFormBase, and the DbHandler service used
   * to list and optimize database tables.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed configuration manager service.
   * @param \Drupal\optimize_database_tables\Service\DbHandler $dbHandler
   *   The database handler service for listing and optimizing tables.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    DbHandler $dbHandler,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
    $this->dbHandler = $dbHandler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('optimize_database_tables.handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return static::OPTIMIZE_DATABASE_SETTINGS;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(
    array $form,
    FormStateInterface $form_state,
  ): array {
    $config = $this->config(self::OPTIMIZE_DATABASE_SETTINGS);

    $tables = $this->dbHandler->getTablesList();

    $form['all_tables'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Optimize all tables'),
      '#default_value' => $config->get('all_tables'),
      '#description' => $this->t('Optimize all tables in the database'),
    ];

    $form['table_list'] = [
      '#type' => 'select',
      '#title' => $this->t('Tables in the database'),
      '#options' => $tables,
      '#default_value' => $config->get('table_list'),
      '#description' => $this->t('Selected tables will be optimized'),
      '#multiple' => TRUE,
      '#attributes' => [
        'size' => 17,
      ],
      '#states' => [
        'visible' => [
          ':input[name="all_tables"]' => ['checked' => FALSE],
        ],
        'required' => [
          ':input[name="all_tables"]' => ['checked' => FALSE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(
    array &$form,
    FormStateInterface $form_state,
  ): void {
    $config = $this->config(self::OPTIMIZE_DATABASE_SETTINGS);
    $config
      ->set('all_tables', $form_state->getValue('all_tables'))
      ->set('table_list', $form_state->getValue('table_list'))
      ->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [static::OPTIMIZE_DATABASE_SETTINGS];
  }

}
