<?php

namespace Drupal\optimize_database_tables\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Provides database-related utilities for the Optimize Database Tables module.
 *
 * This service exposes helper methods to list tables, compute their sizes,
 * format sizes for display, optimize tables depending on the current database
 * driver, and safely quote identifiers when placeholders cannot be used.
 */
class DbHandler {

  /**
   * The Drupal database connection service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The logger channel factory.
   *
   * Used to log errors that occur while issuing database queries or
   * performing maintenance operations.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $logger;

  /**
   * Constructs a new DbHandler service instance.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger
   *   The logger channel factory.
   */
  public function __construct(
    Connection $database,
    LoggerChannelFactoryInterface $logger,
  ) {
    $this->database = $database;
    $this->logger = $logger;
  }

  /**
   * Returns the list of base tables available in the current database.
   *
   * The result is made deterministic by ordering at the SQL level when
   * possible and by applying a case-insensitive natural sort as a safety net.
   * System schemas are excluded on PostgreSQL.
   *
   * @return array
   *   An associative array of table names keyed by their names.
   */
  public function getTablesList(): array {
    $tables = [];
    try {
      $driver = $this->database->databaseType();
      switch ($driver) {
        case 'pgsql':
          $result = $this->database->query(
            "SELECT table_schema, table_name\n" .
            "FROM information_schema.tables\n" .
            "WHERE table_type = 'BASE TABLE'\n" .
            "  AND table_schema NOT IN ('pg_catalog','information_schema')\n" .
            'ORDER BY table_schema, table_name'
          )->fetchAll();
          foreach ($result as $row) {
            $table = (string) $row->table_name;
            $tables[$table] = $table;
          }
          break;

        case 'mysql':
        default:
          // SHOW FULL TABLES returns a row per table; we extract the first
          // column to get the table name.
          $query = $this->database->query("SHOW FULL TABLES WHERE Table_type = 'BASE TABLE'");
          $result = $query->fetchAll();
          foreach ($result as $table_name) {
            $table = (string) current((array) $table_name);
            if ($table !== '') {
              $tables[$table] = $table;
            }
          }
          break;
      }
    }
    catch (\Exception $e) {
      // Log and continue by returning an empty list if an error occurs.
      $this->logger->get('optimize_database_tables')->error($e->getMessage());
    }

    if (!empty($tables)) {
      // Guarantee deterministic order regardless of driver nuances.
      ksort($tables, SORT_NATURAL | SORT_FLAG_CASE);
    }

    return $tables;
  }

  /**
   * Optimizes a single table according to the active database driver.
   *
   * On PostgreSQL, runs VACUUM (ANALYZE). On MySQL/MariaDB, runs OPTIMIZE
   * TABLE. The table name is safely quoted for the specific driver.
   *
   * @param string $table_name
   *   The table name to optimize.
   */
  public function optimizeTable(string $table_name): void {
    try {
      $driver = $this->database->databaseType();
      switch ($driver) {
        case 'pgsql':
          $qualified = $this->quoteIdentifier($table_name, $driver);
          $this->database->query("VACUUM (ANALYZE) {$qualified}");
          break;

        case 'mysql':
        default:
          $qualified = $this->quoteIdentifier($table_name, $driver);
          $this->database->query("OPTIMIZE TABLE {$qualified}")->execute();
          break;
      }
    }
    catch (\Exception $e) {
      // Log and return. Optimization errors should not be fatal.
      $this->logger->get('optimize_database_tables')->error($e->getMessage());
    }
  }

  /**
   * Computes the total size of a table (data + indexes) in bytes.
   *
   * For PostgreSQL, uses pg_total_relation_size on a to_regclass() reference
   * to avoid SQL injection and to handle non-existing tables safely.
   * For MySQL/MariaDB, reads sizes from INFORMATION_SCHEMA.TABLES.
   *
   * @param string $table_name
   *   The table name.
   *
   * @return int
   *   The table size in bytes, or 0 if unavailable or on error.
   */
  public function getTableSizeBytes(string $table_name): int {
    try {
      $driver = $this->database->databaseType();
      switch ($driver) {
        case 'pgsql':
          // Use to_regclass(:tbl) for safety. If the table does not exist,
          // to_regclass returns NULL and COALESCE yields 0.
          $res = $this->database->query(
            'SELECT COALESCE(pg_total_relation_size(to_regclass(:tbl)), 0) AS size',
            [':tbl' => $table_name]
          )->fetchField();
          return (int) $res;

        case 'mysql':
        default:
          $schema = $this->database->getConnectionOptions()['database'] ?? '';
          $result = $this->database->query(
            'SELECT COALESCE(DATA_LENGTH,0) + COALESCE(INDEX_LENGTH,0) AS size
             FROM INFORMATION_SCHEMA.TABLES
             WHERE TABLE_SCHEMA = :schema AND TABLE_NAME = :table',
            [':schema' => $schema, ':table' => $table_name]
          )->fetchField();
          return (int) $result;
      }
    }
    catch (\Exception $e) {
      $this->logger->get('optimize_database_tables')->error($e->getMessage());
      return 0;
    }
  }

  /**
   * Computes the total size in bytes for a list of tables.
   *
   * @param array $tables
   *   A list of table names (strings).
   *
   * @return int
   *   The sum of sizes in bytes for all provided tables.
   */
  public function getTotalSizeBytes(array $tables): int {
    $total = 0;
    foreach ($tables as $table) {
      $name = is_string($table) ? $table : (string) $table;
      $total += $this->getTableSizeBytes($name);
    }
    return $total;
  }

  /**
   * Formats a byte size to a human-readable string.
   *
   * Uses binary units with a base of 1024 and rounds to the given precision.
   *
   * @param int $bytes
   *   The size in bytes.
   * @param int $precision
   *   The number of decimal places. Defaults to 2.
   *
   * @return string
   *   The formatted size string, e.g., "1.23 MB".
   */
  public function formatBytes(int $bytes, int $precision = 2): string {
    if ($bytes <= 0) {
      return '0 B';
    }
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    $power = (int) floor(log($bytes, 1024));
    $power = min($power, count($units) - 1);
    $value = $bytes / (1024 ** $power);
    return number_format($value, $precision) . ' ' . $units[$power];
  }

  /**
   * Quotes/escapes an identifier (table name) for the specific driver.
   *
   * This avoids SQL injection in contexts where placeholders cannot be used,
   * e.g., for identifiers in DDL or maintenance commands.
   *
   * @param string $identifier
   *   The identifier to quote.
   * @param string $driver
   *   The database driver (e.g., 'pgsql' or 'mysql').
   *
   * @return string
   *   The safely quoted identifier.
   */
  protected function quoteIdentifier(string $identifier, string $driver): string {
    switch ($driver) {
      case 'pgsql':
        $escaped = str_replace('"', '""', $identifier);
        return '"' . $escaped . '"';

      case 'mysql':
      default:
        $escaped = str_replace('`', '``', $identifier);
        return '`' . $escaped . '`';
    }
  }

}
