<?php

namespace Drupal\optimize_database_tables\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\optimize_database_tables\Form\OptimizeDatabaseTablesConfigForm;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Service that orchestrates database table optimization via Drupal batches.
 *
 * Reads configuration to determine which tables to optimize, builds batch
 * operations, and reports results to the messenger service.
 */
class OptimizeDatabase {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * Low-level database handler for optimization operations.
   *
   * @var \Drupal\optimize_database_tables\Service\DbHandler
   */
  protected DbHandler $dbHandler;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Messenger service for user-facing status messages.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * Constructs the OptimizeDatabase service.
   *
   * @param \Drupal\optimize_database_tables\Service\DbHandler $dbHandler
   *   The database handler service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    DbHandler $dbHandler,
    ConfigFactoryInterface $configFactory,
    MessengerInterface $messenger,
  ) {
    $this->configFactory = $configFactory;
    $this->dbHandler = $dbHandler;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('optimize_database_tables.handler'),
      $container->get('config.factory'),
      $container->get('messenger'),
    );
  }

  /**
   * Starts a batch process to optimize selected or all database tables.
   *
   * @return void
   *   No return value. Initializes a Drupal batch and exits.
   */
  public function optimizeDatabase(): void {
    $config = $this->configFactory->get(
      OptimizeDatabaseTablesConfigForm::OPTIMIZE_DATABASE_SETTINGS
    );
    $allTables = $config->get('all_tables');
    if ($allTables) {
      $tables = $this->dbHandler->getTablesList();
    }
    else {
      $tables = $config->get('table_list');
    }

    $beforeTotal = $this->dbHandler->getTotalSizeBytes($tables);

    $operations = [];
    $operations[] = [
      [$this, 'initSizeContext'],
      [$beforeTotal],
    ];

    foreach ($tables as $table) {
      $operations[] = [
        [$this, 'optimizeTable'],
        [$table],
      ];
    }
    batch_set([
      'title' => (string) $this->t('Optimize Database Tables'),
      'init_message' => (string) $this->t('Optimize Database Tables'),
      'operations' => $operations,
      'finished' => [$this, 'endOptimizeBatch'],
    ]);
  }

  /**
   * Optimizes the specified database table and updates the context message.
   *
   * @param string $table
   *   The name of the database table to be optimized.
   * @param array $context
   *   The batch context array to store status information.
   *
   * @return void
   *   No return value. Performs table optimization and updates batch context.
   */
  public function optimizeTable(string $table, array &$context): void {
    $context['message'] = (string) $this->t(
      'Optimize table @table',
      ['@table' => $table]
    );
    try {
      $this->dbHandler->optimizeTable($table);
      $context['results'][$table] = $table;
    }
    catch (\Exception $e) {
      $this->messenger->addError($e->getMessage());
    }
  }

  /**
   * Initializes batch context with total DB size before optimization.
   *
   * @param int $beforeTotal
   *   Total size in bytes of all tables to be optimized.
   * @param array $context
   *   Batch context.
   *
   * @return void
   *   No return value. Stores initial size in the batch context.
   */
  public function initSizeContext(int $beforeTotal, array &$context): void {
    $context['results']['_beforeTotal'] = $beforeTotal;
  }

  /**
   * Finalizes the optimization batch process and handles the results.
   *
   * @param bool $success
   *   Indicates whether the batch process was successful.
   * @param array $results
   *   The results of the batch operations.
   * @param array $operations
   *   Remaining operations, if any, after the batch process.
   *
   * @return void
   *   No return value. Sends summary messages to the messenger service.
   */
  public function endOptimizeBatch($success, $results, $operations): void {
    if ($success) {
      // Extract initial total size and the list of optimized tables.
      $beforeTotal = (int) ($results['_beforeTotal'] ?? 0);
      $tables = [];
      foreach ($results as $key => $value) {
        if ($key === '_beforeTotal') {
          continue;
        }
        $tables[] = is_string($value) ? $value : (string) $value;
      }

      // Calculate after total size and savings.
      $afterTotal = $this->dbHandler->getTotalSizeBytes($tables);
      $saved = max(0, $beforeTotal - $afterTotal);
      $percent = $beforeTotal > 0 ? ($saved / $beforeTotal * 100) : 0;

      $this->messenger->addMessage($this->t(
        'Optimization finished for @count table(s).',
        ['@count' => count($tables)]
      ));
      $this->messenger->addMessage($this->t(
        'Before: @before, After: @after, Saved: @saved (@percent%).',
        [
          '@before' => $this->dbHandler->formatBytes($beforeTotal),
          '@after' => $this->dbHandler->formatBytes($afterTotal),
          '@saved' => $this->dbHandler->formatBytes($saved),
          '@percent' => number_format($percent, 2),
        ]
      ));
    }
  }

}
