<?php

namespace Drupal\optimizely\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

use Drupal\optimizely\Util\AccountId;

/**
 * Implements the form for Account Info.
 */
class AccountSettingsForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'optimizely_account_info';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $settings_form['#theme'] = 'optimizely_account_settings_form';

    $settings_form['optimizely_id'] = [
      '#type' => 'number',
      '#title' => $this->t('Optimizely ID Number'),
      '#default_value' => AccountId::getId(),
      '#description' =>
      $this->t('Your Optimizely account ID. This is the number after "/js/" in the
          Optimizely Tracking Code found in your account on the Optimizely website.'),
      '#size' => 60,
      '#maxlength' => 256,
      '#required' => TRUE,
    ];
    $settings_form['actions'] = ['#type' => 'actions'];
    $settings_form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => 'Submit',
    ];

    // Will be $form in the render array and the template file.
    return $settings_form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    // Store the optimizely account id number.
    $optimizely_id = $form_state->getValue('optimizely_id');
    AccountId::setId($optimizely_id);

    // Clear page cache for entries that are active.
    $projects = \Drupal::entityTypeManager()->getStorage('optimizely')->loadByProperties(['state' => TRUE]);

    $paths = array_merge(...array_values(array_map(function($project) {
      return preg_split('/[\r\n]+/', $project->getPaths(), -1, PREG_SPLIT_NO_EMPTY);
    }, $projects)));

    \Drupal::service('optimizely.cacherefresher')->doRefresh($paths);

    // Inform the administrator that the default project / experiment entry
    // is ready to be enabled.
    $this->messenger()->addStatus($this->t('The default project entry is now ready to be enabled.
      This will apply the default Optimizely project tests sitewide.'));

    // Redirect back to projects listing.
    $form_state->setRedirect('entity.optimizely.collection');
  }

}
