<?php

namespace Drupal\optimizely\Form;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityConfirmFormBase;
use Drupal\Core\Url;
use Drupal\Core\Form\FormStateInterface;

use Drupal\optimizely\Util\CacheRefresher;

/**
 * Builds the form to delete an Optimizely instance.
 */
class OptimizelyDeleteForm extends EntityConfirmFormBase {

  /**
   * Utility to refresh cache.
   *
   * @var \Drupal\optimizely\Util\CacheRefresher
   */
  protected $cacheRefresher;

  /**
   * Constructs a new OptimizelyDeleteForm.
   *
   * @param \Drupal\optimizely\Util\CacheRefresher $cache_refresher
   */
  public function __construct(CacheRefresher $cache_refresher) {
    $this->cacheRefresher = $cache_refresher;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('optimizely.cacherefresher')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to delete %name?', ['%name' => $this->entity->label()]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('entity.optimizely.collection');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Prevent deletion of default project.
    if ($this->entity->id() === 'default') {
      $this->messenger()->addError($this->t('Default project cannot be deleted.'));

      // Return to project listing page.
      $form_state->setRedirectUrl($this->getCancelUrl());
      return;
    }

    $this->entity->delete();
    $this->messenger()->addMessage($this->t('Project %label has been deleted.', ['%label' => $this->entity->label()]));

    // Only clear page cache for entries that are active when deleted.
    if ($this->entity->getState()) {
      $path_array = explode("\r\n", $this->entity->getPaths());
      $this->cacheRefresher->doRefresh($path_array);
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
