<?php

namespace Drupal\optimizely\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\optimizely\Util\PathChecker;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for the Optimizely add and edit forms.
 */
class OptimizelyForm extends EntityForm {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The path checker.
   *
   * @var \Drupal\optimizely\Util\PathChecker
   */
  protected $pathChecker;

  /**
   * Constructs an OptimizelyForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entityTypeManager.
   * @param \Drupal\optimizely\Util\PathChecker $path_checker
   *   The pathMatcher.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    PathChecker $path_checker
    ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->pathChecker = $path_checker;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('optimizely.pathchecker')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $optimizely = $this->entity;
    $optimizely_id = \Drupal::configFactory()->getEditable('optimizely.settings')->get('optimizely_id');

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title'),
      '#maxlength' => 255,
      '#default_value' => $optimizely->label(),
      '#description' => $this->t('Descriptive name for the project entry.'),
      '#required' => TRUE,
      '#disabled' => $optimizely->id() === 'default',
    ];
    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $optimizely->id(),
      '#machine_name' => [
        'exists' => [$this, 'exist'],
      ],
      '#disabled' => !$optimizely->isNew(),
    ];
    $form['code'] = [
      '#type' => 'number',
      '#title' => $this->t('Project code'),
      '#size' => 30,
      '#maxlength' => 255,
      '#default_value' => $optimizely->id() === 'default' ? $optimizely_id : $optimizely->getCode(),
      '#description' => ($optimizely->id() === 'default' && !$optimizely_id) ?
        $this->t('The Optimizely account value has not been set in the
          <a href="@url">Account Info</a> settings form.
          The Optimizely account value is used as
          the project ID for this "default" project entry.',
          ['@url' => Url::fromRoute('optimizely.settings')->toString()]
        ) :
        $this->t('The Optimizely javascript file name used in the snippet
          as provided by the Optimizely website for the project.'),
      '#required' => TRUE,
      '#disabled' => $optimizely->id() === 'default',
    ];
    $form['state'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Project enabled'),
      '#default_value' => $optimizely->getState(),
    ];
    $form['paths'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Path(s)'),
      '#default_value' => $optimizely->getPaths(),
      '#description' => $this->t('Enter the paths where you want to insert the Optimizely
        Snippet. For Example: "/clubs/*" causes the snippet to appear on all pages
        below "/clubs" in the URL but not on the actual "/clubs" page itself.'),
      '#cols' => 100,
      '#rows' => 6,
      '#resizable' => FALSE,
      '#required' => FALSE,
      '#weight' => 40,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $optimizely_id = \Drupal::config('optimizely.settings')->get('optimizely_id');
    $project_code = $form_state->getValue('code');
    $project_id = $form_state->getValue('id');

    // Verify that the code isn't the same as the default code.
    if ($project_id !== 'default' && $optimizely_id == $project_code) {
      $form_state->setErrorByName('code',
        $this->t('The project code (%project_code) is the same as the <a href="@settings-page">default code</a>.',
          [
            '%project_code' => $project_code,
            '@settings-page' => Url::fromRoute('optimizely.settings')->toString(),
          ])
      );
    }

    // Verify that another project isn't using the same code.
    $projects = \Drupal::entityTypeManager()->getStorage('optimizely')->loadByProperties(['code' => $project_code]);

    // Filter out current value.
    $found = array_filter($projects, function ($project) use ($project_id) {
      return $project->id() != $project_id;
    });
    $found = reset($found);

    if ($found) {
      $form_state->setErrorByName('code',
        $this->t('This project code is already in use in the <a href="@link">%project</a> project.',
          [
            '%project' => $found->label(),
            '@link' => Url::fromRoute('entity.optimizely.edit', ['optimizely' => $found->id()])->toString,
          ])
      );
    }

    // Verify no duplicate paths.
    if ($form_state->getValue('state')) {

      // Confirm that the project paths point to valid site URLs.
      $target_paths = preg_split('/[\r\n]+/', $form_state->getValue('paths'), -1, PREG_SPLIT_NO_EMPTY);

      $invalid_path = $this->pathChecker->validatePaths($target_paths);

      if (!is_bool($invalid_path)) {
        $form_state->setErrorByName('paths',
          $this->t('The project path "%project_path" could not be resolved as a valid URL for the site,
             or it contains a wildcard * that cannot be handled by this module.',
            ['%project_path' => $invalid_path]));
      }

      // There must be only one Optimizely javascript call on a page.
      // Check paths to ensure there are no duplicates.
      $error_path = $this->pathChecker->uniquePaths($target_paths, $project_id);

      if (!is_bool($error_path)) {
        $form_state->setErrorByName('paths',
          $this->t('The path "%error_path" will result in a duplicate entry based on
            the other project path settings. Optimizely does not allow more
            than one project to be run on a page.',
            ['%error_path' => $error_path]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $project = $this->entity;
    $status = $project->save();

    if ($status === SAVED_NEW) {
      $this->messenger()->addMessage($this->t('Project %label created.', [
        '%label' => $project->label(),
      ]));
    }
    else {
      $this->messenger()->addMessage($this->t('Project %label updated.', [
        '%label' => $project->label(),
      ]));
    }

    // Clear page cache for entries that are active.
    $paths = preg_split('/[\r\n]+/', $project->getPaths(), -1, PREG_SPLIT_NO_EMPTY);

    \Drupal::service('optimizely.cacherefresher')->doRefresh($paths);

    $form_state->setRedirect('entity.optimizely.collection');
  }

  /**
   * Helper function to check whether an Optimizely configuration entity exists.
   */
  public function exist($id) {
    $entity = $this->entityTypeManager->getStorage('optimizely')->getQuery()
      ->accessCheck(TRUE)
      ->condition('id', $id)
      ->execute();
    return (bool) $entity;
  }

  /**
   * Ensure that the path starts with a slash.
   *
   * @param string $path
   *   Path to be checked for having a leading slash. If leading slash
   *   is missing, prefix one. If the path already starts with a special char
   *   such as * or < leave it alone.
   *
   * @return string
   *   The path with a leading slash added, or the original path unchanged.
   */
  private function checkPathLeadingSlash($path) {
    return (ctype_alnum($path[0])) ? '/' . $path : $path;
  }

}
