<?php

namespace Drupal\optimizely\Util;

use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\path_alias\AliasManagerInterface;

/**
 * Implements methods for looking up path aliases and system paths.
 */
class LookupPath {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The path alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $pathAliasManager;

  /**
   * Constructs a LookupPath object.
   *
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language Manager.
   * @param \Drupal\path_alias\AliasManagerInterface $path_alias_manager
   *   The path alias manager.
   */
  public function __construct(
    LanguageManagerInterface $language_manager,
    AliasManagerInterface $path_alias_manager) {
    $this->languageManager = $language_manager;
    $this->pathAliasManager = $path_alias_manager;
  }

  /**
   * Helper function to lookup a path alias, given a path.
   *
   * This function acts as an adapter and passes back a return value
   * like those of drupal_lookup_path(), which has been removed
   * as of Drupal 8.
   */
  public function lookupPathAlias($path) {
    $languages = $this->languageManager->getLanguages();
    $path = $this->checkPath($path);

    foreach ($languages as $language) {
      $alias = $this->pathAliasManager->getAliasByPath($path, $language->getId());
      if ((strcmp($alias, $path) !== 0)) {
        return $alias;
      }
    }

    return FALSE;
  }

  /**
   * Helper function to lookup a system path, given a path alias.
   *
   * This function acts as an adapter and passes back a return value
   * like those of drupal_lookup_path(), which has been removed
   * as of Drupal 8.
   */
  public function lookupSystemPath($path) {
    $languages = $this->languageManager->getLanguages();
    $path = $this->checkPath($path);

    foreach ($languages as $language) {
      $alias = $this->pathAliasManager->getAliasByPath($path, $language->getId());
      if ((strcmp($alias, $path) !== 0)) {
        return $alias;
      }
    }

    return FALSE;
  }

  /**
   * Ensure that $path starts with a forward slash.
   *
   * The alias_manager requires it.
   */
  public function checkPath($path) {
    return ($path[0] == '/') ? $path : '/' . $path;
  }

}
