<?php

namespace Drupal\Tests\optimizely\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Create users with no, some, and optimizely permissions.
 *
 * Use those users to test access to module related pages.
 *
 * @group Optimizely
 */
class OptimizelyAccessTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The path to the listing page.
   *
   * @var string
   */
  protected $listingPage = 'admin/config/system/optimizely';

  /**
   * The path to the addition page.
   *
   * @var string
   */
  protected $addUpdatePage = 'admin/config/system/optimizely/add';

  /**
   * The path to the deletion page.
   *
   * @var string
   */
  protected $deletePage = 'admin/config/system/optimizely/default/delete';

  /**
   * The path to the settings page.
   *
   * @var string
   */
  protected $settingsPage = 'admin/config/system/optimizely/settings';

  /**
   * A user with no additional permissions.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $noPermissionsUser;

  /**
   * A user with permissions to view, add and edit content.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $somePermissionsUser;

  /**
   * A user who can administer Optimizely and view/edit content.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $privilegedUser;

  /**
   * The permission to configure Optimizely.
   *
   * @var string
   */
  protected $optimizelyPermission = 'administer optimizely';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['optimizely', 'node'];

  /**
   * {@inheritdoc}
   */
  public static function getInfo() {
    return [
      'name' => 'Optimizely Access',
      'description' => 'Test that no part of the Optimizely module administration' .
      ' interface can be accessed without the necessary permissions.',
      'group' => 'Optimizely',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {

    parent::setUp();

    $this->drupalCreateContentType(['type' => 'page', 'name' => 'Basic page']);

    $this->noPermissionsUser = $this->drupalCreateUser([]);

    $this->somePermissionsUser = $this->drupalCreateUser([
      'access content',
      'create page content',
      'edit own page content',
    ]);

    // Create an admin user. The user will have the privilege
    // 'administer optimizely'. This privilege is needed to access all
    // administration functionality of the module.
    $this->privilegedUser = $this->drupalCreateUser([
      'access content',
      'create page content',
      'edit own page content',
      // 'administer url aliases',
      // 'create url aliases',.
      $this->optimizelyPermission,
    ]);

  }

  /**
   * Test that the Optimizely permission itself is valid.
   */
  public function testOptimizelyPermission() {

    $valid = $this->checkPermissions(['name' => $this->optimizelyPermission]);
    $this->assertTrue($valid);
  }

  /**
   * Test access to module functionality by users without permission.
   */
  public function testUserNoPermission() {

    $this->checkNoAccess($this->noPermissionsUser);
    $this->checkNoAccess($this->somePermissionsUser);

  }

  /**
   * Check that a user does not have access to the Optimizely pages.
   */
  private function checkNoAccess($user) {

    $access_forbidden = '403';

    $this->drupalLogin($user);

    $this->drupalGet($this->listingPage);
    $this->assertSession()->statusCodeEquals($access_forbidden);

    $this->drupalGet($this->addUpdatePage);
    $this->assertSession()->statusCodeEquals($access_forbidden);

    $this->drupalGet($this->deletePage);
    $this->assertSession()->statusCodeEquals($access_forbidden);

    $this->drupalGet($this->settingsPage);
    $this->assertSession()->statusCodeEquals($access_forbidden);

    $this->drupalLogout();

  }

  /**
   * Test access allowed to module functionality by user with permission.
   */
  public function testUserWithPermission() {

    $access_ok = '200';

    $this->drupalLogin($this->privilegedUser);

    $this->drupalGet($this->listingPage);
    $this->assertSession()->statusCodeEquals($access_ok);

    $this->drupalGet($this->addUpdatePage);
    $this->assertSession()->statusCodeEquals($access_ok);

    $this->drupalGet($this->deletePage);
    $this->assertSession()->statusCodeEquals($access_ok);

    $this->drupalGet($this->settingsPage);
    $this->assertSession()->statusCodeEquals($access_ok);

    $this->drupalLogout();

  }

}
