<?php

namespace Drupal\Tests\optimizely\Functional;

use Drupal\Core\Language\LanguageInterface;
use Drupal\Tests\BrowserTestBase;

/**
 * Test adding a project with a path that is an alias.
 *
 * @group Optimizely
 */
class OptimizelyAddUpdateTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The path to the addition page.
   *
   * @var string
   */
  protected $addUpdatePage = 'admin/config/system/optimizely/add';

  /**
   * The path to edit the second configuration instance.
   *
   * @var string
   */
  protected $update2Page = 'admin/config/system/optimizely/add_update/2';

  /**
   * The path to add an alias.
   *
   * @var string
   */
  protected $addAliasPage = 'admin/config/search/path/add';

  /**
   * A user who can administer Optimizely, view/edit content and add aliases.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $privilegedUser;

  /**
   * The permission to configure Optimizely.
   *
   * @var string
   */
  protected $optimizelyPermission = 'administer optimizely';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['optimizely', 'node', 'language', 'path'];

  /**
   * {@inheritdoc}
   */
  public static function getInfo() {
    return [
      'name' => 'Optimizely Add / Update Project',
      'description' => 'Ensure that the add / update features function properly.',
      'group' => 'Optimizely',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {

    parent::setUp();

    $this->drupalCreateContentType(['type' => 'page', 'name' => 'Basic page']);

    // Create an admin user. The user will have the privilege
    // 'administer optimizely'. This privilege is needed to access all
    // administration functionality of the module.
    $this->privilegedUser = $this->drupalCreateUser([
      'access content',
      'create page content',
      'edit own page content',
      'administer url aliases',
      'create url aliases',
      $this->optimizelyPermission,
    ]);

  }

  /**
   * Test adding and updating a project.
   */
  public function testAddUpdateProject() {

    $this->drupalLogin($this->privilegedUser);

    // N.B. Do NOT use randomString() to generate string values because the
    // resulting strings may contain special chars that break the SQL
    // statements as well as possibly causing other problems.
    // Use randomMachineName() instead since it generates letters and
    // numbers only.
    //
    // ----- create page.
    $settings = [
      'type' => 'page',
      'title' => $this->randomMachineName(32),
      'langcode' => LanguageInterface::LANGCODE_NOT_SPECIFIED,
      'body' => [
                  [
                    'value' => $this->randomMachineName(64),
                    'format' => filter_default_format(),
                  ],
      ],
    ];
    $node1 = $this->drupalCreateNode($settings);

    // Create the url alias.
    $edit_node1 = [];
    $edit_node1['path'] = '/node/' . $node1->id();
    $edit_node1['alias'] = '/' . $this->randomMachineName(10);
    $this->drupalGet($this->addAliasPage);
    $this->submitForm($edit_node1, 'Save');

    // Add a project with a path to the alias.
    $project_title = $this->randomMachineName(8);
    $edit = [
      'id' => $project_title,
      'label' => $project_title,
      'code' => rand(0, 10000),
      'paths' => $edit_node1['alias'],
      'state' => rand(0, 1),
    ];
    $this->drupalGet($this->addUpdatePage);
    $this->submitForm($edit, 'Save');

    $optimizely_config = \Drupal::entityTypeManager()
      ->getStorage('optimizely')
      ->load($project_title);

    $this->assertEquals($optimizely_config->label(), $edit['label']);

    // ----- create page.
    $settings_2 = [
      'type' => 'page',
      'title' => $this->randomMachineName(32),
      'langcode' => LanguageInterface::LANGCODE_NOT_SPECIFIED,
      'body' => [
                  [
                    'value' => $this->randomMachineName(64),
                    'format' => filter_default_format(),
                  ],
      ],
    ];
    $node2 = $this->drupalCreateNode($settings_2);

    // Create another url alias.
    $edit_node2 = [];
    $edit_node2['path'] = '/node/' . $node2->id();
    $edit_node2['alias'] = '/' . $this->randomMachineName(10);
    $this->drupalGet($this->addAliasPage);
    $this->submitForm($edit_node2, 'Save');

    // Update the existing project with the other alias.
    $project_title = $this->randomMachineName(8);
    $edit_2 = [
      'id' => $project_title,
      'label' => $project_title,
      'code' => rand(0, 10000),
      'paths' => $edit_node1['alias'],
      'state' => rand(0, 1),
    ];
    $this->drupalGet($this->update2Page);
    $this->submitForm($edit_2, 'Save');

    // Test if database was updated.
    $optimizely_config = \Drupal::entityTypeManager()
      ->getStorage('optimizely')
      ->load($project_title);

    $this->assertEquals($optimizely_config->label(), $edit_2['label']);
  }

}
