<?php

namespace Drupal\Tests\optimizely\Functional;

use Drupal\Core\Language\LanguageInterface;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests pages that have aliases.
 *
 * Tests that the javascript snippet is included on project pages
 * through aliases and not included on non-project pages through aliases.
 *
 * @group Optimizely
 */
class OptimizelyAliasIncludeSnippetTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The path to the addition page.
   *
   * @var string
   */
  protected $addUpdatePage = 'admin/config/system/optimizely/add';

  /**
   * The path to add an alias.
   *
   * @var string
   */
  protected $addAliasPage = 'admin/config/search/path/add';

  /**
   * A user who can administer Optimizely, view/edit content and add aliases.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $privilegedUser;

  /**
   * The permission to configure Optimizely.
   *
   * @var string
   */
  protected $optimizelyPermission = 'administer optimizely';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['optimizely', 'node', 'path'];

  /**
   * {@inheritdoc}
   */
  public static function getInfo() {
    return [
      'name' => 'Optimizely Alias Include Snippet',
      'description' => 'Ensure that the Optimizely snippet is included in project path when using aliases.',
      'group' => 'Optimizely',
    ];

  }

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {

    parent::setUp();

    $this->drupalCreateContentType(['type' => 'page', 'name' => 'Basic page']);

    $this->privilegedUser = $this->drupalCreateUser([
      'access content',
      'create page content',
      'administer url aliases',
      'create url aliases',
      $this->optimizelyPermission,
    ]);

  }

  /**
   * Test inclusion and non-inclusion of snippet.
   */
  public function testIncludeSnippet() {

    $this->drupalLogin($this->privilegedUser);

    $node1 = $this->makePage();
    $node2 = $this->makePage();
    $node3 = $this->makePage();
    $node4 = $this->makePage();

    $alias1 = $this->makePageAlias($node1);
    $alias2 = $this->makePageAlias($node2);
    $alias3 = $this->makePageAlias($node3);
    $alias4 = $this->makePageAlias($node4);

    // Array holding project field values.
    $project_title = $this->randomMachineName(8);
    $edit = [
      'id' => $project_title,
      'label' => $project_title,
      'code' => rand(0, 10000),
      'paths' => $alias1 . "\n" . $alias2,
      'state' => rand(0, 1),
    ];

    // Create snippet.
    $snippet = '//cdn.optimizely.com/js/' . $edit['code'] . '.js';

    // Create the project.
    $this->drupalGet($this->addUpdatePage);
    $this->submitForm($edit, 'Save');

    // Log out to make sure cache refreshing works.
    $this->drupalLogout();
    // @todo check how to turn on "cache pages for anonymous users"
    // and "Aggregate JavaScript files" on Performance page
    // Check if snippet does appears on project path pages thru alias
    $this->drupalGet("node/" . $node1->id());
    $this->assertSession()->responseNotContains($snippet);

    $this->drupalGet("node/" . $node2->id());
    $this->assertSession()->responseNotContains($snippet);

    // Check if snippet does not appear on other non-project pages.
    $this->drupalGet("node/" . $node3->id());
    $this->assertSession()->responseNotContains($snippet);

    $this->drupalGet("node/" . $node4->id());
    $this->assertSession()->responseNotContains($snippet);

  }

  /**
   * Make a page with a random title.
   */
  private function makePage() {

    $settings = [
      'type' => 'page',
      'title' => $this->randomMachineName(32),
      'langcode' => LanguageInterface::LANGCODE_NOT_SPECIFIED,
      'body' => [
                  [
                    'value' => $this->randomMachineName(64),
                    'format' => filter_default_format(),
                  ],
      ],
    ];
    $node = $this->drupalCreateNode($settings);
    return $node;
  }

  /**
   * Make a random alias to an existing page.
   */
  private function makePageAlias($node) {

    $edit_node = [];
    $edit_node['path'] = '/node/' . $node->id();
    $edit_node['alias'] = '/' . $this->randomMachineName(10);
    $this->drupalGet($this->addAliasPage);
    $this->submitForm($edit_node, 'Save');
    // @todo create alias in 'node/add/page'

    return $edit_node['alias'];

  }

}
