<?php

namespace Drupal\Tests\optimizely\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Test various aspects of the default project.
 *
 * 1. The default project is available but disabled in the project listing page
 *    after the module has been enabled.
 *
 * 2. A message in the project listing page directs the administrator to go
 *    to the module settings page to enter the Optimizely account value.
 *
 * 3. Accessing the account setting page should be blank by default with
 *    a message informing the user that the account setting will be used
 *    for the default project number.
 *
 * 4. Test adding the account setting redirects to the project listing page
 *    with the account number listed as the disabled project dumber for the
 *    default project entry.
 *
 * 5. The default project cannot be enabled until the account number is entered
 *    on the settings page.
 *
 * 6. Enabling the default project with the default path setting of sidewide "*"
 *    should result in the snippet being displayed on the site's front page.
 *
 * @group Optimizely
 */
class OptimizelyDefaultProjectTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The path to the settings page.
   *
   * @var string
   */
  protected $settingsPage = 'admin/config/system/optimizely/settings';

  /**
   * The path to the listing page.
   *
   * @var string
   */
  protected $listingPage = 'admin/config/system/optimizely';

  protected $updateDefaultProjPage = 'admin/config/system/optimizely/add_update/1';

  protected $ajaxCallbackUrl = 'ajax/optimizely';

  /**
   * The permission to configure Optimizely.
   *
   * @var string
   */
  protected $optimizelyPermission = 'administer optimizely';

  /**
   * A user who can administer Optimizely, view/edit content and add aliases.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $privilegedUser;

  protected $optimizelyAccountId;

  /**
   * List of modules to enable.
   *
   * @var array
   */
  protected static $modules = ['optimizely'];

  /**
   * {@inheritdoc}
   */
  public static function getInfo() {

    return [
      'name' => 'Optimizely Default Project',
      'description' => 'Test the existence of a disabled default project.
         When it is enabled after adding the Optimizely account ID,
         the default snippet is added to the front page (default) of the site.',
      'group' => 'Optimizely',
    ];

  }

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {

    parent::setUp();

    $this->privilegedUser = $this->drupalCreateUser([$this->optimizelyPermission]);
  }

  /**
   * Test various aspects of the Default project.
   *
   * 1. The default project is available but disabled in the project listing
   *    page after the module has been enabled.
   *
   * 2. A message in the project listing page directs the administrator to go
   *    to the module settings page to enter the Optimizely account value.
   *
   * 5. The default project cannot be enabled until the account number is
   *    entered on the settings page.
   */
  public function testDefaultProjectEnable() {

    // Access with privileged user.
    $this->drupalLogin($this->privilegedUser);

    // Look for entry in project listing page.
    $this->drupalGet($this->listingPage);

    $this->assertSession()->responseContains(
      '<td class="project-title-column disabled">Default</td>');

    // Confirm default project is not enabled.
    $this->assertSession()->responseContains(
      '<input id="project-enable-1" name="project-1" type="checkbox" ' .
      'value="1" class="form-checkbox" />');
    $this->assertSession()->checkboxNotChecked('project-enable-1');

    // Link to complete default project setup available.
    $this->assertSession()->linkExists('Account Info', 0,
      '<strong>Link from default project to module settings page available.</strong>');

    // Navigate to Edit form for Default project.
    $this->drupalGet($this->updateDefaultProjPage);

    // Title field set to Default, not accessible.
    $this->assertSession()->responseMatches(
      ':<input( [^>]*? | )disabled="disabled"( [^>]*? | )id="edit-optimizely-project-title"( [^>]*? | )value="Default"( [^>]*?)?/>:',
      '<strong>Project title field is not editable and set to "Default"</strong>.');

    // Project Code field not set (Undefined), not accessible.
    $this->assertSession()->responseMatches(
      ':<input( [^>]*? | )disabled="disabled"( [^>]*? | )id="edit-optimizely-project-code"( [^>]*? | )value="Undefined"( [^>]*?)?/>:',
      '<strong>Project code field is not editable and set to "Undefined".</strong>');

    // Link to settings page to set account / Default project code.
    $this->assertSession()->linkExists('Account Info', 0,
      '<strong>Link to settings page found to set Default project code.</strong>');

    // Check default Default project path is set to sitewide wild card.
    $this->assertSession()->responseMatches(
      ':<textarea( [^>]*? | )id="edit-optimizely-path"( [^>]*?)?>\*</textarea>:');

    // * 5. The default project can not be enabled
    // until the account number is entered on the settings page.
  }

  /**
   * Test settings for Default project.
   *
   * 3. Accessing the account setting page should be blank by default with
   *    a message informing the user that the account setting will be used
   *    for the default project number.
   *
   * 4. Test that adding the account setting redirects to the project listing
   *    page with the account number listed as the disabled project dumber for
   *    the default project entry.
   */
  public function testDefaultProjectSettings() {

    // Access with privileged user.
    $this->drupalLogin($this->privilegedUser);

    // Access generate module settings page.
    $this->drupalGet($this->settingsPage);

    // Check for blank setting (default)
    $this->assertSession()->fieldValueEquals('optimizely_id', '');

    // Add Optimizely account setting.
    $this->optimizelyAccountId = rand(1000000, 9999999);
    // N.B. Must use name attribute, not Id.
    $edit = [
      'optimizely_id' => $this->optimizelyAccountId,
    ];
    $this->drupalGet($this->settingsPage);
    $this->submitForm($edit, 'Submit');

    // Check that redirect to project page worked after entering
    // Optimizely account ID in setting page
    // $this->assertUrl('/admin/config/system/optimizely', $options = array(),
    // 'Redirected to project listing page -> /admin/config/system/optimizely
    // after submitting Optimizely account ID on setting page.');.
    $this->drupalGet($this->listingPage);

    // Check that the newly entered Optimizely ID is now listed
    // as the project ID for the Default project.
    $this->assertSession()->responseContains(
      '<td class="project-code-column disabled">' . $this->optimizelyAccountId . '</td>',
      '<strong>Default project is using the Optimizely account ID for project ID -> ' .
      $this->optimizelyAccountId . '.</strong>');

    // Access add / edit project page for default project.
    $this->drupalGet($this->updateDefaultProjPage);

    // Check the project ID setting matches the Optimizely Account ID setting.
    $this->assertSession()->fieldValueEquals('optimizely_project_code', $this->optimizelyAccountId);

    // Enable the Default project.
    $edit = [
      'optimizely_enabled' => 1,
    ];
    $this->drupalGet($this->updateDefaultProjPage);
    $this->submitForm($edit, 'Update');

    // Go to project listings page.
    $this->drupalGet($this->listingPage);

    // Confirm default project *is* enabled.
    $this->assertSession()->responseContains(
      '<input id="project-enable-1" name="project-1" checked="checked" ' .
      'type="checkbox" value="1" class="form-checkbox" />');

  }

  /**
   * Test use of Ajax to enable Default project.
   */
  public function testDefaultProjectListingAjax() {

    // Access with privileged user.
    $this->drupalLogin($this->privilegedUser);

    // Add Optimizely account setting so that Default Project can be enabled.
    $this->optimizelyAccountId = rand(1000000, 9999999);
    // N.B. Must use name attribute, not Id.
    $edit = [
      'optimizely_id' => $this->optimizelyAccountId,
    ];
    $this->drupalGet($this->settingsPage);
    $this->submitForm($edit, 'Submit');

    // Go to project listings page.
    $this->drupalGet($this->listingPage);

    // Confirm default project is disabled.
    $this->assertSession()->checkboxNotChecked('project-enable-1');

    // Test that Ajax call succeeds. 1 == Default Project.
    $params = [
      'target_oid' => 1,
      'target_enable' => 1,
    ];

    $json_response = $this->drupalPost($this->ajaxCallbackUrl, 'application/json', $params);
    $resp_obj = json_decode($json_response);

    $this->assertEquals($resp_obj->status, 'updated');
    $this->assertEquals($resp_obj->message, '');

  }

}
