<?php

namespace Drupal\Tests\optimizely\Functional;

use Drupal\Core\Language\LanguageInterface;
use Drupal\Tests\BrowserTestBase;

/**
 * Test enabling / disabling non-default project from update page.
 *
 * @group Optimizely
 */
class OptimizelyEnableDisableTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The path to the addition page.
   *
   * @var string
   */
  protected $addUpdatePage = 'admin/config/system/optimizely/add';

  /**
   * The path to edit the second configuration instance.
   *
   * @var string
   */
  protected $update2Page = 'admin/config/system/optimizely/add_update/2';

  /**
   * The path to add an alias.
   *
   * @var string
   */
  protected $addAliasPage = 'admin/config/search/path/add';

  /**
   * A user who can administer Optimizely, view/edit content and add aliases.
   *
   * @var string
   */
  protected $privilegedUser;

  /**
   * The permission to configure Optimizely.
   *
   * @var string
   */
  protected $optimizelyPermission = 'administer optimizely';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['optimizely', 'node', 'path'];

  /**
   * {@inheritdoc}
   */
  public static function getInfo() {
    return [
      'name' => 'Optimizely Enable / Disable Project',
      'description' => 'Test enabling / disabling non-default projects.',
      'group' => 'Optimizely',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {

    parent::setUp();

    $this->drupalCreateContentType(['type' => 'page', 'name' => 'Basic page']);

    $this->privilegedUser = $this->drupalCreateUser([
      'access content',
      'create page content',
      'administer url aliases',
      'create url aliases',
      $this->optimizelyPermission,
    ]);
  }

  /**
   * Test enabling and disable on the project update page.
   */
  public function testEnableDisable() {

    $this->drupalLogin($this->privilegedUser);

    // ----- create page.
    $settings = [
      'type' => 'page',
      'title' => $this->randomMachineName(32),
      'langcode' => LanguageInterface::LANGCODE_NOT_SPECIFIED,
      'body' => [
                  [
                    'value' => $this->randomMachineName(64),
                    'format' => filter_default_format(),
                  ],
      ],
    ];
    $node = $this->drupalCreateNode($settings);

    // Create the url alias
    // N.B. The source and alias paths MUST start with a leading slash.
    $edit_node = [];
    $edit_node['path'] = '/node/' . $node->id();
    $edit_node['alias'] = '/' . $this->randomMachineName(10);
    $this->drupalGet($this->addAliasPage);
    $this->submitForm($edit_node, 'Save');

    // Add a project with a path to the alias.
    $project_title = $this->randomMachineName(8);
    $edit = [
      'id' => $project_title,
      'label' => $project_title,
      'code' => rand(0, 10000),
      'paths' => $edit_node['alias'],
      'state' => 1,
    ];
    $this->drupalGet($this->addUpdatePage);
    $this->submitForm($edit, 'Add');

    $edit_2 = [
      'optimizely_enabled' => 1,
    ];
    $this->drupalGet($this->update2Page);
    $this->submitForm($edit_2, 'Update');

    // Test if project was enabled.
    $enabled = \Drupal::database()->query('SELECT enabled FROM {optimizely} WHERE oid = 2')->fetchField();
    $this->assertEquals($enabled, $edit_2['optimizely_enabled']);

    $edit_3 = [
      'optimizely_enabled' => 0,
    ];
    $this->drupalGet($this->update2Page);
    $this->submitForm($edit_3, 'Update');

    // Test if project was disabled.
    $enabled = \Drupal::database()->query('SELECT enabled FROM {optimizely} WHERE oid = 2')->fetchField();
    $this->assertEquals($enabled, $edit_3['optimizely_enabled']);

  }

}
