<?php

namespace Drupal\Tests\optimizely\Functional;

use Drupal\Core\Language\LanguageInterface;
use Drupal\Tests\BrowserTestBase;

/**
 * Test that the javascript snippet is included on project pages.
 *
 * And not included on non-project pages.
 *
 * @group Optimizely
 */
class OptimizelyIncludeSnippetTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The path to the addition page.
   *
   * @var string
   */
  protected $addUpdatePage = 'admin/config/system/optimizely/add';

  /**
   * A user who can administer Optimizely and view/edit content.
   *
   * @var string
   */
  protected $privilegedUser;

  /**
   * The permission to configure Optimizely.
   *
   * @var string
   */
  protected $optimizelyPermission = 'administer optimizely';

  /**
   * List of modules to enable.
   *
   * @var array
   */
  protected static $modules = ['optimizely', 'node'];

  /**
   * {@inheritdoc}
   */
  public static function getInfo() {
    return [
      'name' => 'Optimizely Include Snippet',
      'description' => 'Ensure that the Optimizely snippet is included' .
      ' in project path when not using aliases.',
      'group' => 'Optimizely',
    ];

  }

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {

    parent::setUp();

    $this->drupalCreateContentType(['type' => 'page', 'name' => 'Basic page']);

    $this->privilegedUser = $this->drupalCreateUser([
      'access content',
      'create page content',
      $this->optimizelyPermission,
    ]);

  }

  /**
   * Test whether snippet is correctly included or not included.
   */
  public function testIncludeSnippet() {

    $this->drupalLogin($this->privilegedUser);

    $node1 = $this->makePage();
    $node2 = $this->makePage();
    $node3 = $this->makePage();
    $node4 = $this->makePage();

    // Array holding project field values.
    $project_title = $this->randomMachineName(8);
    $edit = [
      'id' => $project_title,
      'label' => $project_title,
      'code' => rand(0, 10000),
      'paths' => "/node/$node1->id()\n/node/$node2->id()",
      'state' => 1,
    ];

    // Create snippet.
    $snippet = '//cdn.optimizely.com/js/' . $edit['code'] . '.js';

    // Create the project.
    $this->drupalGet($this->addUpdatePage);
    $this->submitForm($edit, 'Save');

    // Log out to make sure cache refreshing works.
    $this->drupalLogout();
    // @todo check how to turn on "cache pages for anonymous users"
    // and "Aggregate JavaScript files" on Performance page
    // Check if snippet does appears on project path pages
    $this->drupalGet("node/" . $node1->id());
    $this->assertSession()->responseNotContains($snippet);

    $this->drupalGet("node/" . $node2->id());
    $this->assertSession()->responseNotContains($snippet);

    // Check if snippet does not appear on other non-project pages.
    $this->drupalGet("node/" . $node3->id());
    $this->assertSession()->responseNotContains($snippet);

    $this->drupalGet("node/" . $node4->id());
    $this->assertSession()->responseNotContains($snippet);

  }

  /**
   * Make a page with a random title.
   */
  private function makePage() {

    $settings = [
      'type' => 'page',
      'title' => $this->randomMachineName(32),
      'langcode' => LanguageInterface::LANGCODE_NOT_SPECIFIED,
      'body' => [
                  [
                    'value' => $this->randomMachineName(64),
                    'format' => filter_default_format(),
                  ],
      ],
    ];
    $node = $this->drupalCreateNode($settings);
    return $node;
  }

}
