<?php

namespace Drupal\orchestration_eca\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca\Plugin\FormFieldYamlTrait;
use Drupal\eca\Service\YamlParser;
use Drupal\orchestration\Webhooks;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Exception\ParseException;

/**
 * Describes the orchestration dispatch_webhook action.
 */
#[Action(
  id: 'orchestration_dispatch_webhook',
  label: new TranslatableMarkup('Dispatch webhook'),
)]
#[EcaAction(
  description: new TranslatableMarkup('Dispatches a webhook and sends extra data along with it.'),
  version_introduced: '1.0.0',
)]
class Webhook extends ConfigurableActionBase {

  use FormFieldYamlTrait;

  /**
   * The webhooks service.
   *
   * @var \Drupal\orchestration\Webhooks
   */
  protected Webhooks $webhooks;

  /**
   * The YAML parser.
   *
   * @var \Drupal\eca\Service\YamlParser
   */
  protected YamlParser $yamlParser;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->webhooks = $container->get('orchestration.webhooks');
    $instance->yamlParser = $container->get('eca.service.yaml_parser');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    $result = AccessResult::allowedIf($this->currentUser->hasPermission('use orchestration'));
    if ($result->isAllowed() && $this->configuration['use_yaml'] && $this->configuration['validate_yaml']) {
      try {
        $this->yamlParser->parse($this->configuration['data']);
      }
      catch (ParseException) {
        $result = AccessResult::forbidden('YAML data is not valid.');
      }
    }
    return $return_as_object ? $result : $result->isAllowed();
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    if ($this->configuration['use_yaml']) {
      try {
        $data = $this->yamlParser->parse($this->configuration['data']);
      }
      catch (ParseException) {
        $this->logger->error('Tried parsing a value in action "orchestration_dispatch_webhook" as YAML format, but parsing failed.');
        return;
      }
    }
    else {
      $data = $this->tokenService->getOrReplace($this->configuration['data']);
    }
    $this->webhooks->dispatch($this->configuration['webhook_id'], $data);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'webhook_id' => '',
      'data' => '',
      'use_yaml' => FALSE,
      'validate_yaml' => FALSE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['webhook_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Webhook'),
      '#description' => $this->t('Select a webhook to dispatch.'),
      '#default_value' => $this->configuration['webhook_id'],
      '#options' => $this->webhooks->getWebhooksForSelect(),
    ];
    $form['data'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Data'),
      '#default_value' => $this->configuration['data'],
      '#weight' => -40,
      '#eca_token_replacement' => TRUE,
    ];
    $this->buildYamlFormFields(
      $form,
      $this->t('Interpret above data as YAML format'),
      $this->t('Nested data can be set using YAML format, for example <em>my_key: "My value"</em>. When using this format, this option needs to be enabled. When using tokens and YAML altogether, make sure that tokens are wrapped as a string. Example: <em>title: "[node:title]"</em>'),
      -30,
    );
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['webhook_id'] = $form_state->getValue('webhook_id');
    $this->configuration['data'] = $form_state->getValue('data');
    $this->configuration['use_yaml'] = !empty($form_state->getValue('use_yaml'));
    $this->configuration['validate_yaml'] = !empty($form_state->getValue('validate_yaml'));
    parent::submitConfigurationForm($form, $form_state);
  }

}
