<?php

namespace Drupal\orchestration_tool;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\orchestration\Service;
use Drupal\orchestration\ServiceConfig;
use Drupal\orchestration\ServicesProviderInterface;
use Drupal\tool\Tool\ToolManager;

/**
 * Provides Tool API tools for the orchestration module.
 */
class ServicesProvider implements ServicesProviderInterface {

  public function __construct(
    protected ToolManager $toolManager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getId(): string {
    return 'tool';
  }

  /**
   * {@inheritdoc}
   */
  public function getAll(): array {
    $services = [];
    /** @var \Drupal\tool\Tool\ToolDefinition $definition */
    foreach ($this->toolManager->getDefinitions() as $id => $definition) {
      $service = new Service(
        $this,
        $id,
        $definition->getLabel(),
        $definition->getDescription(),
      );
      /** @var \Drupal\tool\TypedData\InputDefinition $input */
      foreach ($definition->getInputDefinitions() as $key => $input) {
        $service->addConfig(new ServiceConfig(
          $key,
          $input->getLabel(),
          $input->getDescription() ?? '',
          $input->isRequired(),
          $input->getDataType(),
          !$input->isLocked(),
          $input->getDefaultValue(),
          0,
          $input->getConstraints(),
        ));
      }
      $services[] = $service;
    }
    return $services;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(Service $service, array $config): array|string {
    /** @var \Drupal\tool\Tool\ToolInterface $executableTool */
    $executableTool = $this->toolManager->createInstance($service->id());
    /** @var \Drupal\tool\TypedData\InputDefinition $input */
    foreach ($executableTool->getInputDefinitions() as $key => $input) {
      if (!isset($config[$key])) {
        continue;
      }
      $value = $config[$key];
      if (str_starts_with($input->getDataType(), 'entity:')) {
        $value = $this->entityTypeManager->getStorage(substr($input->getDataType(), 7))->load($value);
      }
      $executableTool->setInputValue($key, $value);
    }
    $executableTool->execute();
    return $executableTool->getResultMessage();
  }

}
