<?php

namespace Drupal\orchestration_ai_agents;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\Enum\AiModelCapability;
use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai_agents\PluginInterfaces\AiAgentInterface;
use Drupal\ai_agents\PluginManager\AiAgentManager;
use Drupal\ai_agents\Task\Task;
use Drupal\Component\Serialization\Exception\InvalidDataTypeException;
use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\orchestration\Service;
use Drupal\orchestration\ServiceConfig;
use Drupal\orchestration\ServicesProviderInterface;

/**
 * Provides AI Agents as services for the orchestration module.
 */
class ServicesProvider implements ServicesProviderInterface {

  public function __construct(
    protected AiAgentManager $aiAgentManager,
    protected AiProviderPluginManager $aiProviderManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected LoggerChannelInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getId(): string {
    return 'ai_agent';
  }

  /**
   * {@inheritdoc}
   */
  public function getAll(): array {
    $services = [];
    $providers = $this->aiProviderManager->getSimpleProviderModelOptions('chat', TRUE, TRUE, [AiModelCapability::ChatJsonOutput]);
    /** @var \Drupal\ai_agents\Entity\AiAgent $agent */
    foreach ($this->entityTypeManager->getStorage('ai_agent')->loadMultiple() as $agent) {
      if (!$agent->status()) {
        continue;
      }
      $service = new Service(
        $this,
        $agent->id(),
        $agent->label(),
        $agent->get('description'),
      );
      $service->addConfig(new ServiceConfig(
        'instructions',
        'Instructions',
        $agent->get('description'),
        TRUE,
      ));
      $service->addConfig(new ServiceConfig(
        'model',
        'Model',
        '',
        TRUE,
        'string',
        TRUE,
        '',
        0,
        ['Choice' => ['choices' => $providers]],
      ));
      $services[] = $service;
    }
    return $services;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(Service $service, array $config): array|string {
    $agent = NULL;
    $error = NULL;
    try {
      $provider = $this->aiProviderManager->loadProviderFromSimpleOption($config['model']);
      /** @var \Drupal\ai_agents\PluginInterfaces\ConfigAiAgentInterface $agent */
      $agent = $this->aiAgentManager->createInstance($service->id());
      $modelName = $this->aiProviderManager->getModelNameFromSimpleOption($config['model']);
      $agent->setAiProvider($provider);
      $agent->setModelName($modelName);
      $agent->setAiConfiguration([]);
      $agent->setCreateDirectly(TRUE);
    }
    catch (\Exception $e) {
      $error = 'Cannot initialize agent: ' . $e->getMessage();
    }
    if ($agent !== NULL) {
      if (!$agent->isAvailable()) {
        $error = 'Agent is not available.';
      }
      else {
        $agent->setTask(new Task($config['instructions']));
        try {
          $solvability = $agent->determineSolvability();
        }
        catch (\Exception $e) {
          $error = $e->getMessage();
          $solvability = AiAgentInterface::JOB_NOT_SOLVABLE;
        }
        switch ($solvability) {
          case AiAgentInterface::JOB_SOLVABLE:
            try {
              $result = [
                'message' => $this->prepareMessage($agent->solve()),
                'services' => [],
              ];
              foreach ($agent->getToolResults(TRUE) as $toolResult) {
                if ($toolResult instanceof ExecutableFunctionCallInterface) {
                  $result['tools'][] = [
                    'name' => $toolResult->getPluginDefinition()['name'],
                    'message' => $this->prepareMessage($toolResult->getReadableOutput()),
                  ];
                }
              }
            }
            catch (\Exception $e) {
              $error = $e->getMessage();
            }
            break;

          case AiAgentInterface::JOB_NEEDS_ANSWERS:
            $error = 'Agent needs more information to complete this task';
            break;

          default:
            $error = $error ?? 'Agent cannot solve this task';
        }
      }
    }
    // Build response.
    if ($error === NULL) {
      $response = [
        'success' => TRUE,
        'result' => $result ?? [],
      ];
      $this->logger->info('Agent "@agent" executed: prompt = "@prompt" → result = @result', [
        '@agent' => $service->id(),
        '@prompt' => $config['instructions'] ?? '(none)',
        '@result' => empty($result) ? '(empty array)' : json_encode($result),
      ]);
    }
    else {
      $response = [
        'success' => FALSE,
        'error' => $error,
      ];
      $this->logger->error('Agent "@agent" failed: prompt = "@prompt" → error = @error', [
        '@agent' => $service->id(),
        '@prompt' => $config['instructions'] ?? '(none)',
        '@error' => $error,
      ]);
    }

    return $response;
  }

  /**
   * Tries decoding the message as YAML or JSON.
   *
   * @param string $message
   *   The message.
   *
   * @return array|string
   *   An array, if the message could be decoded, or the original message as a
   *   string.
   */
  private function prepareMessage(string $message): array|string {
    try {
      return Yaml::decode($message);
    }
    catch (InvalidDataTypeException) {
      try {
        return json_decode($message, TRUE, 512, JSON_THROW_ON_ERROR);
      }
      catch (\JsonException) {
        // We ignore this. Message may neither contain YAML nor JSON.
      }
    }
    return $message;
  }

}
