<?php

namespace Drupal\orchestration_eca\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\orchestration\Event\PollEventTimestamp;

/**
 * Describes the orchestration add_item_to_poll_result_timestamp action.
 */
#[Action(
  id: 'orchestration_add_item_to_poll_result_timestamp',
  label: new TranslatableMarkup('Add item to poll result by timestamp'),
)]
#[EcaAction(
  description: new TranslatableMarkup('Adds an item together with the creation timestamp to the poll result.'),
  version_introduced: '1.0.0',
)]
class AddItemToPollResultTimestamp extends AddItemToPollResultBase {

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    $result = parent::access($object, $account, TRUE);
    if ($result->isAllowed()) {
      $result = AccessResult::allowedIf($this->event instanceof PollEventTimestamp);
    }
    return $return_as_object ? $result : $result->isAllowed();
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $data = $this->getData();
    $timestamp = trim($this->tokenService->getOrReplace($this->configuration['timestamp']));
    if ($timestamp === '') {
      $timestamp = $this->time->getRequestTime();
    }
    /** @var \Drupal\orchestration\Event\PollEventTimestamp $event */
    $event = $this->event;
    $event->addItem((int) $timestamp, $data);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'timestamp' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['timestamp'] = [
      '#type' => 'textfield',
      '#title' => $this->t('The timestamp when the data was created'),
      '#description' => $this->t('Leave empty to use the current timestamp.'),
      '#default_value' => $this->configuration['timestamp'],
      '#eca_token_replacement' => TRUE,
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['timestamp'] = $form_state->getValue('timestamp');
    parent::submitConfigurationForm($form, $form_state);
  }

}
