<?php

namespace Drupal\orchestration_eca\Plugin\ECA\Event;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Attribute\EcaEvent;
use Drupal\eca\Attribute\Token;
use Drupal\eca\Entity\Objects\EcaEvent as EcaEventObject;
use Drupal\eca\Plugin\ECA\Event\EventBase;
use Drupal\orchestration\Event\PollEventBase;
use Drupal\orchestration\Event\PollEventId;
use Drupal\orchestration\Event\PollEventTimestamp;
use Symfony\Contracts\EventDispatcher\Event;

/**
 * ECA event for the polling event.
 */
#[EcaEvent(
  id: 'orchestration_poll',
  deriver: 'Drupal\orchestration_eca\Plugin\ECA\Event\PollDeriver',
  version_introduced: '1.0.0',
)]
class Poll extends EventBase {

  /**
   * {@inheritdoc}
   */
  public static function definitions(): array {
    return [
      'timestamp' => [
        'label' => 'Poll by timestamp',
        'event_name' => 'orchestration_poll.timestamp',
        'event_class' => PollEventTimestamp::class,
      ],
      'id' => [
        'label' => 'Poll by ID',
        'event_name' => 'orchestration_poll.id',
        'event_class' => PollEventId::class,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'event_id' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['event_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Event ID'),
      '#default_value' => $this->configuration['event_id'],
      '#description' => $this->t('The event ID which needs to be the same that the polling uses to receive the data.'),
      '#required' => TRUE,
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    $this->configuration['event_id'] = $form_state->getValue('event_id');
  }

  /**
   * {@inheritdoc}
   */
  public function generateWildcard(string $eca_config_id, EcaEventObject $ecaEvent): string {
    $configuration = $ecaEvent->getConfiguration();
    return trim($configuration['event_id']);
  }

  /**
   * {@inheritdoc}
   *
   * Verifies if this event is due for the next execution.
   *
   * This event stores the last execution time for each modeller event
   * identified by $id and determines with the given frequency, if and when
   * this same event triggered cron should be executed.
   */
  public static function appliesForWildcard(Event $event, string $event_name, string $wildcard): bool {
    if ($event instanceof PollEventBase) {
      return $event->getWildcard() === $wildcard;
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  #[Token(
    name: 'last_poll',
    description: 'The timestamp when the last poll was executed.',
    classes: [PollEventTimestamp::class]
  )]
  #[Token(
    name: 'last_id',
    description: 'The last ID received during the last poll.',
    classes: [PollEventId::class]
  )]
  public function getData(string $key): mixed {
    if ($this->event instanceof PollEventTimestamp && $key === 'last_poll') {
      return $this->event->getTimestamp();
    }
    if ($this->event instanceof PollEventId && $key === 'last_id') {
      return $this->event->getId();
    }
    return parent::getData($key);
  }

}
