<?php

namespace Drupal\orchestration_eca;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Plugin\DataType\EntityAdapter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\State\StateInterface;
use Drupal\eca\Plugin\DataType\DataTransferObject;
use Drupal\eca\Token\TokenInterface;
use Drupal\eca_base\BaseEvents;
use Drupal\eca_base\Event\ToolEvent;
use Drupal\orchestration\Service;
use Drupal\orchestration\ServiceConfig;
use Drupal\orchestration\ServicesProviderInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Provides ECA tools for the orchestration module.
 */
class ServicesProvider implements ServicesProviderInterface {

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected StateInterface $state,
    protected EventDispatcherInterface $eventDispatcher,
    protected TokenInterface $tokenService,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getId(): string {
    return 'eca';
  }

  /**
   * {@inheritdoc}
   */
  public function getAll(): array {
    $services = [];
    /** @var \Drupal\eca\Entity\EcaStorage $eca_storage */
    $eca_storage = $this->entityTypeManager->getStorage('eca');
    $subscribed = current($this->state->get('eca.subscribed', [])['eca_base.tool'] ?? []);
    if ($subscribed) {
      foreach ($subscribed as $eca_id => $wildcards) {
        /** @var \Drupal\eca\Entity\Eca|null $eca */
        $eca = $eca_storage->load($eca_id);
        if (!$eca) {
          // If an ECA model got deleted, we may end up here and then ignore
          // this model as it no longer exists.
          continue;
        }
        foreach ($wildcards as $eca_event_id => $wildcard) {
          if (!($ecaEvent = $eca->getEcaEvent($eca_event_id))) {
            continue;
          }
          $service = new Service(
            $this,
            $wildcard,
            $ecaEvent->getLabel(),
            $ecaEvent->getConfiguration()['description'] ?? '',
          );
          $arguments = Yaml::decode($ecaEvent->getConfiguration()['arguments']) ?? [];
          foreach ($arguments as $name => $argument) {
            $service->addConfig(new ServiceConfig(
              $name,
              $argument['label'],
              $argument['description'],
              $argument['required'] ?? FALSE,
            ));
          }
          $services[] = $service;
        }
      }
    }
    return $services;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(Service $service, array $config): array|string {
    $names = [];
    foreach ($config as $name => $context) {
      $names[] = $name;
      $this->tokenService->addTokenData($name, $context);
    }
    $event = new ToolEvent($service->id());
    $event->addTokenNamesFromString(implode(',', $names));
    $this->eventDispatcher->dispatch($event, BaseEvents::TOOL);
    $output = $event->getOutput();
    if ($output instanceof EntityAdapter) {
      $output = $output->getEntity();
    }
    if ($output instanceof DataTransferObject) {
      $output = $output->getValue();
    }
    if ($output instanceof EntityInterface) {
      $output = $output->toArray();
    }
    if ($output instanceof FieldItemListInterface) {
      $output = $output->getValue();
    }
    if (is_scalar($output)) {
      $result = (string) $output;
    }
    elseif ($output === NULL) {
      $result = 'undefined';
    }
    else {
      $result = (array) $output;
    }
    return $result;
  }

}
