<?php

namespace Drupal\orchestration\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\orchestration\Event\PollEventId;
use Drupal\orchestration\Event\PollEventTimestamp;
use Drupal\orchestration\ServicesProviderManager;
use Drupal\orchestration\Webhooks;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides routes for the connect module.
 */
final class Connect extends ControllerBase {

  /**
   * Constructs a new Connect object.
   */
  public function __construct(
    protected Request $request,
    protected EventDispatcherInterface $eventDispatcher,
    protected Webhooks $webhooks,
    protected ServicesProviderManager $servicesProviderManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): Connect {
    return new Connect(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('event_dispatcher'),
      $container->get('orchestration.webhooks'),
      $container->get('orchestration.services_manager'),
    );
  }

  /**
   * Provides a list of all available services with their configuration forms.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function services(): JsonResponse {
    return new JsonResponse(array_values($this->servicesProviderManager->getAllServices()));
  }

  /**
   * Executes a service.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response.
   */
  public function execute(): JsonResponse {
    $data = json_decode($this->request->getContent(), TRUE);
    try {
      return new JsonResponse($this->servicesProviderManager->executeService($data['id'] ?? '', $data['config'] ?? []));
    }
    catch (\Exception $e) {
      return new JsonResponse(['error' => $e->getMessage()], 500);
    }
  }

  /**
   * Register a webhook.
   *
   * Once a webhook is registered, it will be stored and can be dispatched.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response.
   */
  public function webhookRegister(): JsonResponse {
    $data = json_decode($this->request->getContent(), TRUE);
    $this->webhooks->register($data['id'], $data['webHookUrl']);
    return new JsonResponse($data);
  }

  /**
   * Unregister a webhook.
   *
   * The stored webhook will be removed and can no longer be dispatched.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response.
   */
  public function webhookUnregister(): JsonResponse {
    $data = json_decode($this->request->getContent(), TRUE);
    $this->webhooks->unregister($data['id']);
    return new JsonResponse($data);
  }

  /**
   * Provides a route to poll the Drupal site.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response.
   */
  public function poll(): JsonResponse {
    $data = json_decode($this->request->getContent(), TRUE);
    if (isset($data['timestamp'])) {
      $event = new PollEventTimestamp($data['name'], $data['timestamp']);
      $eventName = 'orchestration_poll.timestamp';
    }
    elseif (isset($data['id'])) {
      $event = new PollEventId($data['name'], $data['id']);
      $eventName = 'orchestration_poll.id';
    }
    else {
      return new JsonResponse(['error' => 'No timestamp or id provided.'], 400);
    }
    $this->eventDispatcher->dispatch($event, $eventName);
    return new JsonResponse($event->getOutput());
  }

}
