<?php

namespace Drupal\orchestration;

/**
 * A generic service implementation for the orchestration module.
 */
final class Service implements \JsonSerializable {

  /**
   * The configuration fields for this service.
   *
   * @var \Drupal\orchestration\ServiceConfig[]
   */
  protected array $config = [];

  /**
   * Constructs the service.
   */
  public function __construct(
    protected ServicesProviderInterface $provider,
    protected string $id,
    protected string $label,
    protected string $description,
  ) {}

  /**
   * Gets the unique ID of this service.
   *
   * @return string
   *   A unique ID for this service.
   */
  public function uuid(): string {
    return $this->provider->getId() . '::' . $this->id;
  }

  /**
   * Gets the service ID.
   *
   * @return string
   *   The service ID.
   */
  public function id(): string {
    return $this->id;
  }

  /**
   * Gets the service's label.
   *
   * @return string
   *   The service's label.
   */
  public function label(): string {
    return $this->label;
  }

  /**
   * Add a configuration field to this service.
   *
   * @param \Drupal\orchestration\ServiceConfig $config
   *   The configuration field.
   *
   * @return \Drupal\orchestration\Service
   *   The service.
   */
  public function addConfig(ServiceConfig $config): Service {
    $this->config[] = $config;
    return $this;
  }

  /**
   * Converts the service to an array.
   *
   * @return array
   *   The service as an array.
   */
  public function __serialize(): array {
    return [
      'id' => $this->uuid(),
      'label' => $this->label,
      'description' => $this->description,
      'config' => $this->sortedConfig(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function jsonSerialize(): array {
    return $this->__serialize();
  }

  /**
   * Gets the provider of this service.
   *
   * @return \Drupal\orchestration\ServicesProviderInterface
   *   The provider of this service.
   */
  public function getProvider(): ServicesProviderInterface {
    return $this->provider;
  }

  /**
   * Helper function to sort config fields by weight.
   *
   * @return array
   *   The sorted config fields.
   */
  private function sortedConfig(): array {
    $config = $this->config;
    usort($config, function (ServiceConfig $a, ServiceConfig $b) {
      return $a->getWeight() - $b->getWeight();
    });
    return $config;
  }

}
