<?php

namespace Drupal\orchestration;

/**
 * A generic config field implementation for services of orchestration.
 */
final class ServiceConfig implements \JsonSerializable {

  /**
   * Constructs the service.
   */
  public function __construct(
    protected string $key,
    protected string $label,
    protected string $description,
    protected bool $required,
    protected string $type = 'string',
    protected bool $isEditable = TRUE,
    protected mixed $defaultValue = '',
    protected int $weight = 0,
    protected array $constraints = [],
  ) {}

  /**
   * Converts the config field to an array.
   *
   * @return array
   *   The config field as an array.
   */
  final public function __serialize(): array {
    return [
      'key' => $this->key,
      'label' => $this->label,
      'description' => $this->description,
      'required' => $this->required,
      'type' => $this->type,
      'editable' => $this->isEditable,
      'default_value' => $this->defaultValue,
      'weight' => $this->weight,
      'options' => $this->getOptionsFromConstraints(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function jsonSerialize(): array {
    return $this->__serialize();
  }

  /**
   * Extracts options from constraints.
   *
   * @return array
   *   The options.
   */
  private function getOptionsFromConstraints(): array {
    $options = [];
    if (isset($this->constraints['Choice']['choices'])) {
      foreach ($this->constraints['Choice']['choices'] as $constraint => $name) {
        if (is_int($constraint)) {
          $constraint = $name;
        }
        $options[] = [
          'key' => $constraint,
          'name' => $name,
        ];
      }
    }
    return $options;
  }

  /**
   * Gets the weight of this config field.
   *
   * @return int
   *   The weight.
   */
  public function getWeight(): int {
    return $this->weight;
  }

}
