<?php

declare(strict_types=1);

namespace Drupal\orchestration;

use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Provides a collection of webhook services.
 */
final class Webhooks {

  use StringTranslationTrait;

  /**
   * The key value store collection.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueStoreInterface
   */
  protected KeyValueStoreInterface $collection;

  /**
   * Constructs a Webhooks object.
   */
  public function __construct(
    protected readonly KeyValueFactoryInterface $keyvalue,
  ) {
    $this->collection = $this->keyvalue->get('orchestration');
  }

  /**
   * Register a webhook.
   *
   * @param string $id
   *   The webhook ID.
   * @param string $url
   *   The webhook URL.
   */
  public function register(string $id, string $url): void {
    $webhooks = $this->collection->get('webhooks', []);
    $webhooks[$id] = $url;
    $this->collection->set('webhooks', $webhooks);
  }

  /**
   * Unregister a webhook.
   *
   * @param string $id
   *   The webhook ID.
   */
  public function unregister(string $id): void {
    $webhooks = $this->collection->get('webhooks', []);
    unset($webhooks[$id]);
    $this->collection->set('webhooks', $webhooks);
  }

  /**
   * Dispatch a webhook.
   *
   * @param string $id
   *   The webhook ID.
   * @param string|array $data
   *   The data to be posted to the webhook.
   */
  public function dispatch(string $id, string|array $data): void {
    $webhooks = $this->collection->get('webhooks', []);
    $url = $webhooks[$id] ?? NULL;
    if ($url === NULL) {
      return;
    }

    $client = new Client();
    try {
      $options = [
        'headers' => [
          'Content-Type' => 'application/json',
        ],
        'body' => is_scalar($data) ? $data : json_encode($data, JSON_THROW_ON_ERROR),
      ];
      $client->request('POST', $url, $options);
    }
    catch (GuzzleException | \JsonException) {
      // @todo Log this exception.
    }
  }

  /**
   * Provides a select list with all registered webhooks.
   *
   * @return array
   *   The select list with all registered webhooks.
   */
  public function getWebhooksForSelect(): array {
    $webhooks = ['' => $this->t('- Select -')];
    foreach (array_keys($this->collection->get('webhooks', [])) as $id) {
      $webhooks[$id] = $id;
    }
    return $webhooks;
  }

}
