<?php

namespace Drupal\orchestration;

/**
 * Manages all the services providers.
 */
class ServicesProviderManager {

  /**
   * The collected orchestration services providers.
   *
   * @var \Drupal\orchestration\ServicesProviderInterface[]
   */
  protected array $servicesProviders = [];

  /**
   * The available services.
   *
   * @var \Drupal\orchestration\Service[]
   */
  protected array $services;

  /**
   * Returns all available services.
   *
   * @return \Drupal\orchestration\Service[]
   *   All available services.
   */
  public function getAllServices(): array {
    if (!isset($this->services)) {
      $this->services = [];
      foreach ($this->servicesProviders as $serviceProvider) {
        foreach ($serviceProvider->getAll() as $service) {
          $this->services[$service->uuid()] = $service;
        }
      }
    }
    uasort($this->services, function (Service $a, Service $b) {
      return strcmp($a->label(), $b->label());
    });
    return $this->services;
  }

  /**
   * Adds a services provider to the manager.
   *
   * This method is called by the service collector for each tagged service.
   *
   * @param \Drupal\orchestration\ServicesProviderInterface $servicesProvider
   *   The services provider to add.
   */
  public function addServicesProvider(ServicesProviderInterface $servicesProvider): void {
    $this->servicesProviders[$servicesProvider->getId()] = $servicesProvider;
  }

  /**
   * Executes the service with given parameters.
   *
   * @param string $serviceId
   *   The service ID.
   * @param array $config
   *   The service configuration.
   *
   * @return array|string
   *   The execution result.
   */
  public function executeService(string $serviceId, array $config): array|string {
    $this->getAllServices();
    $service = $this->services[$serviceId] ?? NULL;
    if ($service === NULL) {
      throw new \InvalidArgumentException(sprintf('Service %s not found.', $serviceId));
    }
    return $service->getProvider()->execute($service, $config);
  }

}
