<?php

namespace Drupal\orchestration_ai_function;

use Drupal\ai\Service\FunctionCalling\FunctionCallPluginManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\orchestration\Service;
use Drupal\orchestration\ServiceConfig;
use Drupal\orchestration\ServicesProviderInterface;

/**
 * Provides AI Function Calls as services for the orchestration module.
 */
class ServicesProvider implements ServicesProviderInterface {

  public function __construct(
    protected FunctionCallPluginManager $functionCallManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModuleHandler $moduleHandler,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getId(): string {
    return 'ai_function';
  }

  /**
   * {@inheritdoc}
   */
  public function getAll(): array {
    $services = [];
    $ecaInstalled = $this->moduleHandler->moduleExists('orchestration_eca');
    $aiAgentsInstalled = $this->moduleHandler->moduleExists('orchestration_ai_agents');
    $toolInstalled = $this->moduleHandler->moduleExists('orchestration_tool');
    foreach ($this->functionCallManager->getDefinitions() as $id => $definition) {
      if ($ecaInstalled && $definition['provider'] === 'eca_base') {
        // Avoid duplicate services.
        continue;
      }
      if ($aiAgentsInstalled && $definition['provider'] === 'ai_agents') {
        // Avoid duplicate services.
        continue;
      }
      if ($toolInstalled && str_starts_with($definition['provider'], 'tool')) {
        // Avoid duplicate services.
        continue;
      }
      $service = new Service(
        $this,
        $id,
        $definition['name'],
        $definition['description'] ?? '',
      );
      /** @var \Drupal\Core\Plugin\Context\ContextDefinitionInterface $context */
      foreach ($definition['context_definitions'] as $key => $context) {
        $service->addConfig(new ServiceConfig(
          $key,
          $context->getLabel(),
          $context->getDescription() ?? '',
          $context->isRequired(),
          $context->getDataType(),
          TRUE,
          $context->getDefaultValue(),
          0,
          $context->getConstraints(),
        ));
      }
      $services[] = $service;
    }
    return $services;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(Service $service, array $config): array|string {
    /** @var \Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface $executableFunctionCall */
    $executableFunctionCall = $this->functionCallManager->createInstance($service->id());
    /** @var \Drupal\Core\Plugin\Context\ContextDefinitionInterface $input */
    foreach ($executableFunctionCall->getPluginDefinition()['context_definitions'] as $key => $input) {
      if (!isset($config[$key])) {
        continue;
      }
      $value = $config[$key];
      if (str_starts_with($input->getDataType(), 'entity:')) {
        $value = $this->entityTypeManager->getStorage(substr($input->getDataType(), 7))->load($value);
      }
      $executableFunctionCall->setContextValue($key, $value);
    }
    $executableFunctionCall->execute();
    return $executableFunctionCall->getReadableOutput();
  }

}
