<?php

namespace Drupal\orchestration_eca\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\orchestration\Event\PollEventId;

/**
 * Describes the orchestration add_item_to_poll_result_id action.
 */
#[Action(
  id: 'orchestration_add_item_to_poll_result_id',
  label: new TranslatableMarkup('Add item to poll result by ID'),
)]
#[EcaAction(
  description: new TranslatableMarkup('Adds an item together with its ID to the poll result.'),
  version_introduced: '1.0.0',
)]
class AddItemToPollResultId extends AddItemToPollResultBase {

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    $result = parent::access($object, $account, TRUE);
    if ($result->isAllowed()) {
      $result = AccessResult::allowedIf($this->event instanceof PollEventId);
    }
    return $return_as_object ? $result : $result->isAllowed();
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $data = $this->getData();
    $id = trim($this->tokenService->getOrReplace($this->configuration['item_id']));
    if ($id === '') {
      $id = '0';
    }
    /** @var \Drupal\orchestration\Event\PollEventId $event */
    $event = $this->event;
    $event->addItem($id, $data);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'item_id' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['item_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('The ID of the item'),
      '#default_value' => $this->configuration['item_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['item_id'] = $form_state->getValue('item_id');
    parent::submitConfigurationForm($form, $form_state);
  }

}
