<?php

namespace Drupal\orejime\services;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\media\OEmbed\ProviderRepositoryInterface;
use Drupal\media\OEmbed\Resource;
use Drupal\media\OEmbed\ResourceFetcher;
use GuzzleHttp\ClientInterface;

/**
 * Orejime resource fetcher.
 */
class OrejimeResourceFetcher extends ResourceFetcher {

  /**
   * The Orejime config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $settings;

  /**
   * Constructs a ResourceFetcher object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\media\OEmbed\ProviderRepositoryInterface $providers
   *   The oEmbed provider repository service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   The cache backend.
   * @param int $timeout
   *   The length of time to wait for the request before the request
   *   should time out.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   A config factory for retrieving required config objects.
   */
  public function __construct(protected ClientInterface $http_client, protected ProviderRepositoryInterface $providers, protected CacheBackendInterface $cache_backend, protected int $timeout, ConfigFactoryInterface $config_factory) {
    parent::__construct($http_client, $providers, $cache_backend, $timeout);
    $this->settings = $config_factory->get('orejime.settings');
  }

  /**
   * {@inheritdoc}
   */
  protected function createResource(array $data, $url): Resource {
    if ($this->settings->get('iframe_consent')) {
      $this->changeUrl($data);
    }
    return parent::createResource($data, $url);
  }

  /**
   * Add Autoplay for YouTube and Vimeo.
   *
   * @param array $data
   *   The resource data returned by the provider.
   */
  private function changeUrl(array &$data): void {
    preg_match('/src="([^"]+)"/', $data['html'], $match);
    if (isset($match[1])) {
      $url = $match[1];
      switch ($data['provider_name']) {
        case 'Vimeo':
        case 'YouTube':
          $url .= '&autoplay=1';
          break;
      }
      $data['html'] = str_replace($match[1], $url, $data['html']);
    }
  }

}
