<?php

namespace Drupal\organization_field\Form;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Menu\MenuLinkManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The config form builds setting up the number of organizations.
 *
 * And checking the status of ROR API.
 *
 * This form stores the number of organizations.
 */
class OrganizationFieldSettingsForm extends ConfigFormBase {
  /**
   * A http client interface.
   */
  protected ClientInterface $httpClient;

  /**
   * A messenger interface.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The cache menu instance.
   */
  protected CacheBackendInterface $cacheMenu;

  /**
   * The menu link manager instance.
   */
  protected MenuLinkManagerInterface $menuLinkManager;

  /**
   * The config factory for the form.
   */
  public function __construct(ConfigFactoryInterface $config_factory,
        MenuLinkManagerInterface $menuLinkManager,
        CacheBackendInterface $cacheMenu,
        ClientInterface $http_client,
        MessengerInterface $messenger) {
    parent::__construct($config_factory);
    $this->cacheMenu = $cacheMenu;
    $this->menuLinkManager = $menuLinkManager;
    $this->httpClient = $http_client;
    $this->messenger = $messenger;
  }

  /**
   * Instantiate interfaces.
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('config.factory'),
          $container->get('plugin.manager.menu.link'),
          $container->get('cache.menu'),
          $container->get('http_client'),
          $container->get('messenger')
      );
  }

  /**
   * Get the config file name.
   */
  protected function getEditableConfigNames() {
    return [
      'organization_field.settings',
    ];
  }

  /**
   * Set up the form ID.
   */
  public function getFormId() {
    return 'organization_field_settings';
  }

  /**
   * Build the form for the number of organizations and API checking routine.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Retrieving the stored config value or setting a default value.
    $config = $this->config('organization_field.settings');
    $default_ror_api = $config->get('ror_api') ?: '';

    // ROR API URL field on its own line for consistent alignment.
    $form['ror_api'] = [
      '#title' => $this->t('ROR API'),
      '#type' => 'textfield',
      '#size' => 80,
      '#default_value' => $default_ror_api,
      '#placeholder' => $this->t('ROR API'),
      '#attributes' => [],
    ];

    // Action button placed below the text field.
    $form['check_api_status'] = [
      '#type' => 'submit',
      '#value' => $this->t('Check API Status'),
      '#weight' => 1,
      '#submit' => [[$this, 'checkApiStatus']],
      '#limit_validation_errors' => [],
      '#attributes' => [],
    ];

    $depth_values = range(1, 20);
    $form['ror_items_depth'] = [
      '#type' => 'select',
      '#title' => $this->t('No of organizations to show in autocomplete'),
      '#description' => $this->t('Maximal items to display inside "Organization Name" field.'),
      '#default_value' => $config->get('ror_items_depth'),
      '#options' => array_combine($depth_values, $depth_values),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Validate values for the form elements.
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if (!UrlHelper::isValid(trim($form_state->getValue('ror_api')), TRUE)) {
      $form_state->setErrorByName('ror_api',
        $this->t('The ROR API, "@rorapi", is not valid.',
          ['@rorapi' => trim($form_state->getValue('ror_api'))]));
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * Accesses the ROR API and displays the message for the search result.
   */
  public function checkApiStatus(array &$form, FormStateInterface $form_state) {
    $ror_api = trim($form_state->getUserInput()['ror_api']);

    try {
      $response2 = $this->httpClient->request('GET', $ror_api . '/0168r3w48');
      $statusCode = $response2->getStatusCode();
      if ($statusCode != 200) {
        $this->messenger->addError(t("
        <b>API service is down </b><br/>
        <b>Website: </b> <a href='https://ror.readme.io/docs/rest-api' target='_blank'>Research Organization Registry</a> <br/>
        "));
      }
      else {
        $json_string = $response2->getBody();

        $this->messenger->addMessage(t('
        <div>
          <b>API service is up and running </b><br/>
          <b>Website: </b> <a href="https://ror.readme.io/docs/rest-api" target="_blank">Research Organization Registry</a> <br/>
          <b>Endpoint URL: </b>
          <a>@ror_api/0168r3w48</a>
          <br/>
          <b>Status Code: </b> @statusCode <br/>
          <a href="#" type="button" onclick="document.getElementById(`content`).style.display = (document.getElementById(`content`).style.display == `none`) ? `block` : `none`" class="collapsible"><b>Show/Hide Response Body </b></a> <br/>

          <div id="content" style="display:none">
            @json_string <br/>
          </div>
        </div>
        ',
          [
            '@ror_api' => $ror_api,
            '@statusCode' => $statusCode,
            '@json_string' => $json_string,
          ]
        ));

        $this->config('organization_field.settings')
          ->set('ror_api', $ror_api)
          ->set('ror_items_depth', $form_state->getUserInput()['ror_items_depth'])
          ->save();
        $messages = $this->t('The configuration options have been saved.');
        $this->messenger->addStatus($messages);
      }

    }
    catch (RequestException $e) {
      // Wrap the markup in a div.
      $messages = $this->t('The ROR API, "@ror_api" is not valid.',
        ['@ror_api' => $ror_api]);
      $this->messenger->addError($messages);
    }
    catch (GuzzleException $e) {
    }
  }

  /**
   * The action function for the form submit.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('organization_field.settings')
      ->set('ror_api', $form_state->getValue('ror_api'))
      ->set('ror_items_depth', $form_state->getValue('ror_items_depth'))
      ->save();
    parent::submitForm($form, $form_state);
    $this->cacheMenu->invalidateAll();
    $this->menuLinkManager->rebuild();
  }

}
