<?php

namespace Drupal\organization_field\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\CronInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\FieldConfigInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Defines a form to uninstall module dependencies.
 *
 * Removes node.organization_field from the database for uninstallation.
 */
class OrganizationFieldUninstallValidatorForm extends ConfigFormBase {
  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager service.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * Cron service.
   */
  protected CronInterface $cronManager;

  /**
   * The item list to be deleted.
   *
   * @var array
   */
  protected array $itemsToDelete = [];

  /**
   * Constructs a new GroupOrganizationFieldUninstallValidator.
   */
  public function __construct(ConfigFactoryInterface $config_factory,
        EntityTypeManagerInterface $entity_type_manager,
        EntityFieldManagerInterface $entity_field_manager,
        CronInterface $cron_manager) {
    parent::__construct($config_factory);
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->cronManager = $cron_manager;
  }

  /**
   * Instantiate services.
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('config.factory'),
          $container->get('entity_type.manager'),
          $container->get('entity_field.manager'),
          $container->get('cron')
      );
  }

  /**
   * Set up the form ID.
   */
  public function getFormId() {
    return 'organization_field_uninstall_settings';
  }

  /**
   * Set up the config file name. It is not used.
   */
  protected function getEditableConfigNames() {
    return [
      'organization_field.uninstall_settings',
    ];
  }

  /**
   * Shows the field list for content types and storage types.
   *
   * That are required to delete the organization field.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $weight = 0;
    $form['title'] = [
      '#markup' => $this->t('<div><b>ORGANIZATION FIELDS</b></div>'),
      '#weight' => $weight++,
    ];
    $form['break1'] = [
      '#markup' => '<hr>',
      '#weight' => $weight++,
    ];

    $entity_type_ids = [
      'node' => 'node_type',
      // User entity has a single bundle with ID 'user'.
      'user' => NULL,
    ];

    foreach ($entity_type_ids as $entity_type_id => $bundle_entity_type_id) {
      $bundle_ids = [];
      if ($bundle_entity_type_id) {
        $bundles = $this->entityTypeManager->getStorage($bundle_entity_type_id)->loadMultiple();
        foreach ($bundles as $bundle) {
          $bundle_ids[] = $bundle->id();
        }
      }
      else {
        $bundle_ids[] = $entity_type_id; // 'user' => 'user'.
      }

      foreach ($bundle_ids as $bundle_id) {
        $fields = array_filter(
          $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id),
          static function ($fieldDefinition) {
            return $fieldDefinition instanceof FieldConfigInterface;
          }
        );

        foreach ($fields as $key => $definition) {
          if ($definition->getType() == 'organization_field') {
            $this->itemsToDelete[$entity_type_id . ':' . $bundle_id][] = $key;
          }
        }
      }
    }

    $body = '<br/><ul>';
    if (!empty($this->itemsToDelete)) {
      foreach ($this->itemsToDelete as $type => $fieldsArray) {
        foreach ($fieldsArray as $fieldName) {
          $body .= "<li> $fieldName ($type) </li>";
        }
      }
    }
    $body .= '</ul>';

    // Create a Markup object for the body content.
    $safeBody = Markup::create($body);

    $form['body'] = [
      '#markup' => Markup::create((string) $safeBody . '<br/>This action cannot be undone.<br/>Make a backup of your database if you want to be able to restore these items.'),
      '#weight' => $weight++,
    ];
    $form['actions']['submit_cancel'] = [
      '#type' => 'submit',
      '#weight' => $weight++,
      '#value' => t('Cancel'),
      '#submit' => [[static::class, 'customFormCancel']],
      '#limit_validation_errors' => [],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Cancel button on the form.
   */
  public static function customFormCancel(array &$form, FormStateInterface $form_state) {
    $form_state->setRedirect('system.modules_uninstall');
  }

  /**
   * On submit, execute delete operation of the fields shown on form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $is_deleted = FALSE;
    if (!empty($this->itemsToDelete)) {
      foreach ($this->itemsToDelete as $entity_and_bundle => $fieldsArray) {
        // Expand the concatenated key: 'entity:bundle'.
        [$entity_id, $bundle_id] = explode(':', $entity_and_bundle, 2);
        foreach ($fieldsArray as $fieldName) {
          $is_deleted = TRUE;
          $field = FieldConfig::loadByName($entity_id, $bundle_id, $fieldName);
          $field?->delete();
        }
      }

      if ($is_deleted) {
        // Run cron job to clean up the fields.
        $this->cronManager->run();
      }

      $this->messenger()->addMessage($this->t('All organization fields deleted'));
    }
    else {
      $this->messenger()->addMessage($this->t('Nothing to delete'));
    }

    $form_state->setRedirect('system.modules_uninstall');

    parent::submitForm($form, $form_state);
  }

}
