<?php

namespace Drupal\organization_field;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleUninstallValidatorInterface;
use Drupal\field\FieldConfigInterface;

/**
 * An organization field module.
 *
 * Uninstall validator finds dependency modules that have this field.
 * If there are any modules, the "remove" button is added into the uninstall summaries.
 */
class OrganizationFieldUninstallValidator implements ModuleUninstallValidatorInterface {
  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity field manager service.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * Constructs a new GroupOrganizationFieldUninstallValidator.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager,
        EntityFieldManagerInterface $entity_field_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
  }

  /**
   * Checks the module dependencies for the content types and storage types.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function validate($module) {
    $reasons = [];
    if ($module == 'organization_field') {
      // Check both content types and the user account bundle.
      $entity_type_ids = [
        'node' => 'node_type',
        // The user entity has a single bundle with ID 'user'.
        'user' => NULL,
      ];
      $items_to_delete = [];
      foreach ($entity_type_ids as $entity_type_id => $bundle_entity_type_id) {
        $bundle_ids = [];
        if ($bundle_entity_type_id) {
          $bundles = $this->entityTypeManager->getStorage($bundle_entity_type_id)->loadMultiple();
          foreach ($bundles as $bundle) {
            $bundle_ids[] = $bundle->id();
          }
        }
        else {
          // Single bundle entity type like 'user'.
          $bundle_ids[] = $entity_type_id;
        }

        foreach ($bundle_ids as $bundle_id) {
          $fields = array_filter(
            $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id),
            static function ($fieldDefinition) {
              return $fieldDefinition instanceof FieldConfigInterface;
            }
          );
          foreach ($fields as $field_name => $definition) {
            if ($definition->getType() == 'organization_field') {
              $items_to_delete[$entity_type_id . ':' . $bundle_id][] = $field_name;
            }
          }
        }
      }
      if (!empty($items_to_delete)) {
        // Link to the cleanup form using a route for robustness.
        $url = \Drupal\Core\Url::fromRoute('organization_field.uninstall_settings')->toString();
        $reasons[] = t("<a href=':url'>Remove fields in order to uninstall this module</a> Note: Runs cron job for deletion", [':url' => $url]);
      }
    }

    return $reasons;
  }

}
