<?php

namespace Drupal\organization_field\Controller;

use Drupal\Component\Utility\Xss;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Fetches data from the ROR REST API.
 *
 * Populates it in the autocomplete organization field.
 *
 * Reference - https://ror.readme.io/docs/rest-api.
 */
class JsonApiController extends ControllerBase {
  /**
   * A http client.
   */
  protected ClientInterface $httpClient;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new CustomWidget object.
   */
  public function __construct(ConfigFactoryInterface $config_factory,
        ClientInterface $http_client) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
          $container->get('config.factory'),
          $container->get('http_client')
      );
  }

  /**
   * Fetches the specified number of the organizations from ROR API.
   */
  public function handleAutocomplete(Request $request): JsonResponse {
    // Organization field autocomplete
    // Returns [value, label] array of results.
    $results = [];
    $input = $request->query->get('q');

    if (!$input) {
      return new JsonResponse($results);
    }
    $input = Xss::filter($input);

    try {
      $config = $this->configFactory->get('organization_field.settings');
      $ror_api = (string) $config->get('ror_api');

      $response = $this->httpClient->request('GET', $ror_api, [
        'query' => ['query' => $input],
      ]);
      $json_string = (string) $response->getBody();
      $json = json_decode($json_string, TRUE);
      if (!is_array($json) || json_last_error() !== JSON_ERROR_NONE) {
        return new JsonResponse($results);
      }

      $no_of_results = (int) ($json['number_of_results'] ?? 0);
      $max_rors_to_display = (int) ($config->get('ror_items_depth') ?? 0);
      if ($max_rors_to_display > 0) {
        $no_of_results = min($no_of_results, $max_rors_to_display);
      }

      foreach (array_slice($json['items'] ?? [], 0, $no_of_results) as $item) {
        foreach (($item['names'] ?? []) as $name) {
          if (
            isset($name['types']) && is_array($name['types']) &&
            in_array('ror_display', $name['types'], true) &&
            isset($name['value']) && $name['value'] !== ''
          ) {
            $results[] = [
              'value' => $name['value'],
              'label' => $name['value'],
            ];
          }
        }
      }

      return new JsonResponse($results);
    }
    catch (\Throwable $e) {
      // Swallow errors and return empty results for robustness.
      return new JsonResponse([]);
    }
  }

}
