<?php

namespace Drupal\organization_field\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the configurable 'Organization Field' formatter.
 *
 * The organization field formatter configures the labels of the organization,
 * link to organization, link to Research organization registry, link types,
 * list styles, etc. It views the organization information
 * based on the configuration for the organization field.
 *
 * @FieldFormatter(
 *   id = "configurable_organization_field_formatter",
 *   module = "organization_field",
 *   label = @Translation("Organization field"),
 *   field_types = {
 *     "organization_field"
 *   }
 * )
 */
class ConfigurableOrganizationFieldFormatter extends FormatterBase {
  /*---------------------------------------------------------------------
   *
   * View.
   *
   *---------------------------------------------------------------------*/

  /**
   * Set up values for form elements to view in the theme template.
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $element = [];
    $this->sanitizeSettings();

    foreach ($items as $delta => $item) {
      $values = $item->getValue();
      $element[$delta] = [
        '#theme' => 'configurable_organization_field_formatter',
        '#attributes' => [
          'class' => [
            'organization_field',
          ],
        ],
      ];

      // Name Settings.
      $element[$delta]['#org_name_label'] = $this->getSetting('orgNameLabel');
      $element[$delta]['#name_label_link'] = $this->getSetting('nameLabelLink');

      // URL Settings.
      $element[$delta]['#show_org_url'] = $this->getSetting('showOrgUrl');
      $element[$delta]['#org_url_label'] = $this->getSetting('orgUrlLabel');
      $element[$delta]['#show_org_url_link'] = $this->getSetting('showOrgUrlLink');
      $element[$delta]['#url_open_link_in'] = $this->getSetting('urlOpenLinkIn');
      $element[$delta]['#org_url_no_referrer'] = $this->getSetting('orgurlnoreferrer');
      $element[$delta]['#org_url_no_opener'] = $this->getSetting('orgurlnoopener');
      $element[$delta]['#org_url_no_follow'] = $this->getSetting('orgurlnofollow');
      $element[$delta]['#url_list_style'] = $this->getSetting('urlListStyle');
      $element[$delta]['#url_list_separator'] = $this->getSetting('urlListSeparator');
      $element[$delta]['#no_of_urls_to_show'] = $this->getSetting('noOfUrlsToShow');

      // ROR Settings.
      $element[$delta]['#show_ror_url'] = $this->getSetting('showRORUrl');
      $element[$delta]['#ror_url_label'] = $this->getSetting('rorUrlLabel');
      $element[$delta]['#ror_label_icon_or_link'] = $this->getSetting('rorLabelIconOrLink');
      $element[$delta]['#ror_open_link_in'] = $this->getSetting('ROROpenLinkIn');
      $element[$delta]['#show_ror_url_link'] = $this->getSetting('showRORUrlLink');
      $element[$delta]['#ror_url_no_referrer'] = $this->getSetting('rorurlnoreferrer');
      $element[$delta]['#ror_url_no_opener'] = $this->getSetting('rorurlnoopener');
      $element[$delta]['#ror_url_no_follow'] = $this->getSetting('rorurlnofollow');

      // Variables.
      $element[$delta]['#organization_name'] = $values['organization_name'];
      if ($values['organization_url']) {
        $element[$delta]['#organization_url'] = explode(',', $values['organization_url']);
      }

      if ($values['organization_ror_id_url']) {
        $element[$delta]['#organization_ror_id_url'] = $values['organization_ror_id_url'];
      }
    }

    return $element;
  }

  /**
   * Set up the default values for form elements in formatter.
   */
  public static function defaultSettings() {
    return array_merge(
          [
            'orgNameLabel' => 'Organization',
            'orgUrlLabel' => 'Links',
            'rorUrlLabel' => 'Organization identifier',
            'showOrgUrl' => TRUE,
            'showOrgUrlLink' => TRUE,
            'nameLabelLink' => 'org',
            'urlOpenLinkIn' => '_self',
            'urlListStyle' => 'span',
            'urlListSeparator' => ', ',
            'noOfUrlsToShow' => '3',
            'orgurlnoreferrer' => FALSE,
            'orgurlnoopener' => FALSE,
            'orgurlnofollow' => FALSE,
            'rorLabelIconOrLink' => 'url',
            'showRORUrl' => TRUE,
            'showRORUrlLink' => TRUE,
            'ROROpenLinkIn' => '_self',
            'rorurlnoreferrer' => FALSE,
            'rorurlnoopener' => FALSE,
            'rorurlnofollow' => FALSE,
          ],
          parent::defaultSettings());
  }

  /**
   * View current summaries for the organization field formatter.
   */
  public function settingsSummary() {
    // Sanitize current settings.
    $this->sanitizeSettings();

    // Summarize.
    $summary = [];

    if ($this->getSetting('orgNameLabel')) {
      $summary[] = $this->t('Labels: @orgNameLabel, @orgUrlLabel, @rorUrlLabel',
      [
        '@orgNameLabel' => $this->getSetting('orgNameLabel'),
        '@orgUrlLabel' => $this->getSetting('orgUrlLabel'),
        '@rorUrlLabel' => $this->getSetting('rorUrlLabel'),
      ]);
    }

    switch ($this->getSetting('nameLabelLink')) {
      case 'org':
        $summary[] = $this->t('Name: Links to organization URL');
        break;

      case 'ror':
        $summary[] = $this->t('Name: Links to ROR URL');
        break;
    }

    if (!$this->getSetting('showOrgUrlLink')) {
      $summary[] = $this->t('URL: No link');
    }
    else {
      switch ($this->getSetting('urlOpenLinkIn')) {
        case '_self':
          $summary[] = $this->t('URL: Open in current tab/window');
          break;

        case '_blank':
          $summary[] = $this->t('URL: Open in new tab/window');
          break;
      }

      $text = '';
      if ($this->getSetting('orgurlnoreferrer')) {
        $text .= 'noreferrer ';
      }

      if ($this->getSetting('orgurlnoopener')) {
        $text .= 'noopener ';
      }

      if ($this->getSetting('orgurlnofollow')) {
        $text .= 'nofollow ';
      }

      if (empty($text) === FALSE) {
        $summary[] = 'URL: ' . $text;
      }
    }

    switch ($this->getSetting('urlListStyle')) {
      case 'span':
        $summary[] = $this->t('URL: Single line list, with separator');
        break;

      case 'ol':
        $summary[] = $this->t('URL: Numbered list');
        break;

      case 'ul':
        $summary[] = $this->t('URL: Bulleted list');
        break;

      case 'div':
        $summary[] = $this->t('URL: Non-bulleted block list');
        break;
    }

    if (empty($this->getSetting('noOfUrlsToShow')) === FALSE) {
      $summary[] = $this->t('URL: Show @noOfUrlsToShow urls',
      ['@noOfUrlsToShow' => $this->getSetting('noOfUrlsToShow')]);
    }
    else {
      $summary[] = $this->t('URL: Show all urls');
    }

    switch ($this->getSetting('rorLabelIconOrLink')) {
      case 'icon':
        $summary[] = $this->t('ROR URL: Icon');
        break;

      case 'url':
        $summary[] = $this->t('ROR URL: URL');
        break;
    }

    if (!$this->getSetting('showRORUrlLink')) {
      $summary[] = $this->t('ROR URL: No link');
    }
    else {
      switch ($this->getSetting('ROROpenLinkIn')) {
        case '_self':
          $summary[] = $this->t('ROR URL: Open in current tab/window');
          break;

        case '_blank':
          $summary[] = $this->t('ROR URL: Open in new tab/window');
          break;
      }

      $text = '';
      if ($this->getSetting('rorurlnoreferrer')) {
        $text .= 'noreferrer ';
      }

      if ($this->getSetting('rorurlnoopener')) {
        $text .= 'noopener ';
      }

      if ($this->getSetting('rorurlnofollow')) {
        $text .= 'nofollow ';
      }

      if (empty($text) === FALSE) {
        $summary[] = 'ROR URL: ' . $text;
      }
    }

    return $summary;
  }

  /*---------------------------------------------------------------------
   *
   * Settings form.
   *
   *---------------------------------------------------------------------*/

  /**
   * Returns an array of choices for how to open a link.
   *
   * @return string[]
   *   Returns an associative array with internal names as keys and
   *   human-readable translated names as values.
   */
  protected static function getOpenLinkInValues() {
    return [
      '_self' => t('Open link in the same tab/window'),
      '_blank' => t('Open link in a new tab/window'),
    ];
  }

  /**
   * Returns an array of list styles.
   *
   * @return string[]
   *   Returns an associative array with internal names as keys and
   *   human-readable translated names as values.
   */
  protected static function getListStyles() {
    return [
      'span' => t('Single line list'),
      'ol' => t('Numbered list'),
      'ul' => t('Bulleted list'),
      'div' => t('Non-bulleted block list'),
    ];
  }

  /**
   * Returns an array of list styles.
   *
   * @return string[]
   *   Returns an associative array with internal names as keys and
   *   human-readable translated names as values.
   */
  protected static function getNameLinkLabels() {
    return [
      'org' => t('Link to the organization'),
      'ror' => t('Link to Research organization registry'),
    ];
  }

  /**
   * Returns an array of list styles.
   *
   * @return string[]
   *   Returns an associative array with internal names as keys and
   *   human-readable translated names as values.
   */
  protected static function getRorLabelOptions() {
    return [
      'icon' => t('Icon next to organization name'),
      'url' => t('URL'),
    ];
  }

  /**
   * Configure the form elements in the formatter.
   */
  public function settingsForm(array $form, FormStateInterface $formState) {
    $this->sanitizeSettings();

    $weight = 0;
    $marker = rand();

    $elements = [];
    $elements['sectionTitle1'] = [
      '#markup' => $this->t('<div><b>ORGANIZATION NAME SETTINGS</b></div>'),
      '#weight' => $weight++,
    ];
    $elements['sectionBreak1'] = [
      '#markup' => '<hr>',
      '#weight' => $weight++,
    ];

    $elements['orgNameLabel'] = [
      '#title' => $this->t('Name Label'),
      '#type' => 'textfield',
      '#size' => 10,
      '#default_value' => $this->getSetting('orgNameLabel'),
      '#weight' => $weight++,
      '#attributes' => [
        'autocomplete' => 'off',
        'autocapitalize' => 'none',
        'spellcheck' => 'false',
        'autocorrect' => 'off',
      ],
      '#wrapper_attributes' => [
        'class' => [
          '',
        ],
      ],
    ];

    $elements['nameLabelLink'] = [
      '#type' => 'radios',
      '#default_value' => $this->getSetting('nameLabelLink'),
      '#weight' => $weight++,
      '#options' => $this->getNameLinkLabels(),
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-org-url-link-show-name-label',
        ],
      ],
      '#attributes' => [
        'class' => [
          'showNameLabelLink-' . $marker,
        ],
      ],
    ];

    $elements['sectionTitle2'] = [
      '#markup' => $this->t('<div><b>ORGANIZATION URL SETTINGS</b></div>'),
      '#weight' => $weight++,
    ];
    $elements['sectionBreak2'] = [
      '#markup' => '<hr>',
      '#weight' => $weight++,
    ];

    $elements['showOrgUrl'] = [
      '#title' => $this->t('Show organization url(s)'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('showOrgUrl'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-org-url-link-show-org-url',
        ],
      ],
      '#attributes' => [
        'class' => [
          'showOrgUrl-' . $marker,
        ],
      ],
    ];

    $elements['orgUrlLabel'] = [
      '#title' => $this->t('URL Label'),
      '#type' => 'textfield',
      '#size' => 10,
      '#default_value' => $this->getSetting('orgUrlLabel'),
      '#weight' => $weight++,
      '#attributes' => [
        'autocomplete' => 'off',
        'autocapitalize' => 'none',
        'spellcheck' => 'false',
        'autocorrect' => 'off',
      ],
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-padding-left-15',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showOrgUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
      ],
    ];

    $elements['showOrgUrlLink'] = [
      '#title' => $this->t('Link to the item'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('showOrgUrlLink'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-padding-left-15',
        ],
      ],
      '#attributes' => [
        'class' => [
          'showOrgUrlLink-' . $marker,
        ],
      ],
      '#states' => [
        'visible' => [
          '.showOrgUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
      ],
    ];

    $elements['urlOpenLinkIn'] = [
      '#title' => $this->t('Use link to'),
      '#type' => 'select',
      '#options' => $this->getOpenLinkInValues(),
      '#default_value' => $this->getSetting('urlOpenLinkIn'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-org-url-link-open-link-in',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showOrgUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
        'disabled' => [
          '.showOrgUrlLink-' . $marker => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    $elements['orgurlnoreferrer'] = [
      '#title' => $this->t('Do not pass the current site as the referrer ("noreferrer")'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('orgurlnoreferrer'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-org-url-link-noreferrer',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showOrgUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
        'disabled' => [
          '.showOrgUrlLink-' . $marker => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    $elements['orgurlnoopener'] = [
      '#title' => $this->t('Do not share the current page context ("noopener")'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('orgurlnoopener'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-org-url-link-noopener',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showOrgUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
        'disabled' => [
          '.showOrgUrlLink-' . $marker => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    $elements['orgurlnofollow'] = [
      '#title' => $this->t('Do not treat the link as an endorsement ("nofollow")'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('orgurlnofollow'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-org-url-link-nofollow',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showOrgUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
        'disabled' => [
          '.showOrgUrlLink-' . $marker => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    $elements['urlListStyle'] = [
      '#title' => $this->t('List style'),
      '#type' => 'select',
      '#options' => $this->getListStyles(),
      '#default_value' => $this->getSetting('urlListStyle'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-padding-left-15',
        ],
      ],
      '#attributes' => [
        'class' => [
          'urlListStyle-' . $marker,
        ],
      ],
      '#states' => [
        'visible' => [
          '.showOrgUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
      ],
    ];

    $elements['urlListSeparator'] = [
      '#title' => $this->t('Separator'),
      '#type' => 'textfield',
      '#size' => 10,
      '#default_value' => $this->getSetting('urlListSeparator'),
      '#weight' => $weight++,
      '#attributes' => [
        'autocomplete' => 'off',
        'autocapitalize' => 'none',
        'spellcheck' => 'false',
        'autocorrect' => 'off',
      ],
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-org-url-list-separator',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showOrgUrl-' . $marker => [
            'checked' => TRUE,
          ],
          '.urlListStyle-' . $marker => [
            'value' => 'span',
          ],
        ],
      ],
    ];

    $elements['noOfUrlsToShow'] = [
      '#title' => $this->t('No of urls to show'),
      '#type' => 'textfield',
      '#size' => 10,
      '#default_value' => $this->getSetting('noOfUrlsToShow'),
      '#weight' => $weight++,
      '#attributes' => [
        'autocomplete' => 'off',
        'autocapitalize' => 'none',
        'spellcheck' => 'false',
        'autocorrect' => 'off',
      ],
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-padding-left-15',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showOrgUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
      ],
    ];

    $elements['sectionTitle4'] = [
      '#markup' => $this->t('<div><b>ROR URL SETTINGS</b></div>'),
      '#weight' => $weight++,
    ];
    $elements['sectionBreak4'] = [
      '#markup' => '<hr>',
      '#weight' => $weight++,
    ];

    $elements['showRORUrl'] = [
      '#title' => $this->t('Show Research Organization Registry'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('showRORUrl'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          '',
        ],
      ],
      '#attributes' => [
        'class' => [
          'showRORUrl-' . $marker,
        ],
      ],
    ];

    $elements['rorUrlLabel'] = [
      '#title' => $this->t('Label'),
      '#type' => 'textfield',
      '#size' => 10,
      '#default_value' => $this->getSetting('rorUrlLabel'),
      '#weight' => $weight++,
      '#attributes' => [
        'autocomplete' => 'off',
        'autocapitalize' => 'none',
        'spellcheck' => 'false',
        'autocorrect' => 'off',
      ],
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-padding-left-15',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showRORUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
      ],
    ];

    $elements['rorLabelIconOrLink'] = [
      '#title' => $this->t('Show as'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('rorLabelIconOrLink'),
      '#weight' => $weight++,
      '#options' => $this->getRorLabelOptions(),
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-padding-radios-15',
        ],
      ],
      '#attributes' => [
        'class' => [
          'showRorLabelIconOrLink-' . $marker,
        ],
      ],
      '#states' => [
        'visible' => [
          '.showRORUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
      ],
    ];

    $elements['showRORUrlLink'] = [
      '#title' => $this->t('Link to the item'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('showRORUrlLink'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-padding-left-15',
        ],
      ],
      '#attributes' => [
        'class' => [
          'showRORUrlLink-' . $marker,
        ],
      ],
      '#states' => [
        'visible' => [
          '.showRORUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
      ],
    ];

    $elements['ROROpenLinkIn'] = [
      '#title' => $this->t('Use link to'),
      '#type' => 'select',
      '#options' => $this->getOpenLinkInValues(),
      '#default_value' => $this->getSetting('ROROpenLinkIn'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-ror-link-open-link-in',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showRORUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
        'disabled' => [
          '.showRORUrlLink-' . $marker => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    $elements['rorurlnoreferrer'] = [
      '#title' => $this->t('Do not pass the current site as the referrer ("noreferrer")'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('rorurlnoreferrer'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-ror-link-noreferrer',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showRORUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
        'disabled' => [
          '.showRORUrlLink-' . $marker => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    $elements['rorurlnoopener'] = [
      '#title' => $this->t('Do not share the current page context ("noopener")'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('rorurlnoopener'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-ror-link-noopener',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showRORUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
        'disabled' => [
          '.showRORUrlLink-' . $marker => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    $elements['rorurlnofollow'] = [
      '#title' => $this->t('Do not treat the link as an endorsement ("nofollow")'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('rorurlnofollow'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-ror-link-nofollow',
        ],
      ],
      '#states' => [
        'visible' => [
          '.showRORUrl-' . $marker => [
            'checked' => TRUE,
          ],
        ],
        'disabled' => [
          '.showRORUrlLink-' . $marker => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    return $elements;
  }

  /**
   * Sanitize current settings to insure that they are safe and valid.
   *
   * @internal
   * Drupal's class hierarchy for plugins and their settings does not
   * include a 'validate' function, like that for other classes with forms.
   * Validation must therefore occur on use, rather than on form submission.
   *
   * @endinternal
   */
  protected function sanitizeSettings() {
    // Get current settings.
    $url_open_link_in = $this->getSetting('urlOpenLinkIn');
    $url_list_style = $this->getSetting('urlListStyle');
    $ror_open_link_in = $this->getSetting('ROROpenLinkIn');

    $showOrgUrl = $this->getSetting('showOrgUrl');
    $showOrgUrlLink = $this->getSetting('showOrgUrlLink');
    $orgurlnoreferrer = $this->getSetting('orgurlnoreferrer');
    $orgurlnoopener = $this->getSetting('orgurlnoopener');
    $orgurlnofollow = $this->getSetting('orgurlnofollow');

    $showRORUrl = $this->getSetting('showRORUrl');
    $showRORUrlLink = $this->getSetting('showRORUrlLink');
    $rorurlnoreferrer = $this->getSetting('rorurlnoreferrer');
    $rorurlnoopener = $this->getSetting('rorurlnoopener');
    $rorurlnofollow = $this->getSetting('rorurlnofollow');

    $nameLabelLink = $this->getSetting('nameLabelLink');
    $noOfUrlsToShow = $this->getSetting('noOfUrlsToShow');
    $rorLabelIconOrLink = $this->getSetting('rorLabelIconOrLink');

    // Get setting defaults.
    $defaults = $this->defaultSettings();

    $openLinkInValues = $this->getOpenLinkInValues();
    if (empty($url_open_link_in) === TRUE || isset($openLinkInValues[$url_open_link_in]) === FALSE) {
      $url_open_link_in = $defaults['urlOpenLinkIn'];
      $this->setSetting('urlOpenLinkIn', $url_open_link_in);
    }

    if (empty($ror_open_link_in) === TRUE || isset($openLinkInValues[$ror_open_link_in]) === FALSE) {
      $ror_open_link_in = $defaults['ROROpenLinkIn'];
      $this->setSetting('ROROpenLinkIn', $ror_open_link_in);
    }

    $listStyles = $this->getListStyles();
    if (empty($url_list_style) === TRUE || isset($listStyles[$url_list_style]) === FALSE) {
      $url_list_style = $defaults['urlListStyle'];
      $this->setSetting('urlListStyle', $url_list_style);
    }

    $nameLinkLabels = $this->getNameLinkLabels();
    if (empty($nameLabelLink) === TRUE || isset($nameLinkLabels[$nameLabelLink]) === FALSE) {
      $name_label_link = $defaults['nameLabelLink'];
      $this->setSetting('nameLabelLink', $name_label_link);
    }

    $rorLabelOptions = $this->getRorLabelOptions();
    if (empty($rorLabelIconOrLink) === TRUE || isset($rorLabelOptions[$rorLabelIconOrLink]) === FALSE) {
      $ror_label_icon_or_link = $defaults['rorLabelIconOrLink'];
      $this->setSetting('rorLabelIconOrLink', $ror_label_icon_or_link);
    }

    if (empty($noOfUrlsToShow) === FALSE && is_numeric($noOfUrlsToShow) === FALSE) {
      $no_of_urls_to_show = $defaults['noOfUrlsToShow'];
      $this->setSetting('noOfUrlsToShow', $no_of_urls_to_show);
    }

    // Insure boolean values are boolean for Organization URL.
    $showOrgUrl = boolval($showOrgUrl);
    $showOrgUrlLink = boolval($showOrgUrlLink);
    $orgurlnoreferrer = boolval($orgurlnoreferrer);
    $orgurlnoopener = boolval($orgurlnoopener);
    $orgurlnofollow = boolval($orgurlnofollow);

    $this->setSetting('showOrgUrl', $showOrgUrl);
    $this->setSetting('showOrgUrlLink', $showOrgUrlLink);
    $this->setSetting('orgurlnoreferrer', $orgurlnoreferrer);
    $this->setSetting('orgurlnoopener', $orgurlnoopener);
    $this->setSetting('orgurlnofollow', $orgurlnofollow);

    // Insure boolean values are boolean for ROR URL.
    $showRORUrl = boolval($showRORUrl);
    $showRORUrlLink = boolval($showRORUrlLink);
    $rorurlnoreferrer = boolval($rorurlnoreferrer);
    $rorurlnoopener = boolval($rorurlnoopener);
    $rorurlnofollow = boolval($rorurlnofollow);

    $this->setSetting('showRORUrl', $showRORUrl);
    $this->setSetting('showRORUrlLink', $showRORUrlLink);
    $this->setSetting('rorurlnoreferrer', $rorurlnoreferrer);
    $this->setSetting('rorurlnoopener', $rorurlnoopener);
    $this->setSetting('rorurlnofollow', $rorurlnofollow);

    if ($showOrgUrl === FALSE) {
      $this->setSetting('showOrgUrlLink', FALSE);
    }
    if ($showRORUrl === FALSE) {
      $this->setSetting('showRORUrlLink', FALSE);
    }
  }

}
