<?php

namespace Drupal\organization_field\Plugin\Field\FieldWidget;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'organization_field_widget' widget.
 *
 * A widget organization renders the organization fields inside forms.
 *
 * @FieldWidget(
 *   id = "organization_field_widget",
 *   module = "organization_field",
 *   label = @Translation("Organization form field with widget"),
 *   field_types = {
 *     "organization_field"
 *   }
 * )
 */
class OrganizationFieldWidget extends WidgetBase {
  /**
   * An http client.
   */
  protected ClientInterface $httpClient;

  /**
   * The configuration factory service.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a new CustomWidget object.
   */
  public function __construct($plugin_id,
  $plugin_definition,
        FieldDefinitionInterface $field_definition,
        array $settings,
        array $third_party_settings,
        ConfigFactoryInterface $config_factory,
        ClientInterface $http_client) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
          $plugin_id,
          $plugin_definition,
          $configuration['field_definition'],
          $configuration['settings'],
          $configuration['third_party_settings'],
          $container->get('config.factory'),
          $container->get('http_client'),
      );
  }

  /**
   * As the default setting, show the organization URL and ROR fields.
   */
  public static function defaultSettings() {
    return array_merge(
          [
            'hideLink' => FALSE,
            'hideROR' => FALSE,
          ],
          parent::defaultSettings());
  }

  /**
   * Summarize the currently selected settings in the summary for the widget.
   */
  public function settingsSummary() {
    $summary = [];

    if ($this->getSetting('hideLink')) {
      $summary[] = 'Hidden: Link field ';
    }

    if ($this->getSetting('hideROR')) {
      $summary[] = 'Hidden: ROR field ';
    }

    if (empty($summary) === TRUE) {
      $summary[] = 'Hidden: None';
    }

    return $summary;
  }

  /**
   * Defines the form to allow for users to change the values for the setting.
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $this->sanitizeSettings();

    $weight = 0;
    $marker = rand();

    $elements['hideLink'] = [
      '#title' => $this->t('Hide link field'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('hideLink'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-padding-left-15',
        ],
      ],
      '#attributes' => [
        'class' => [
          'hideLink-' . $marker,
        ],
      ],
    ];

    $elements['hideROR'] = [
      '#title' => $this->t('Hide ROR URL field'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('hideROR'),
      '#weight' => $weight++,
      '#wrapper_attributes' => [
        'class' => [
          'formatter_suite-padding-left-15',
        ],
      ],
      '#attributes' => [
        'class' => [
          'hideROR-' . $marker,
        ],
      ],
    ];

    return $elements;
  }

  /**
   * Sanitize settings to make sure that they are safe and valid.
   *
   * @internal
   * Drupal's class hierarchy for plugins and their settings does not
   * include a 'validate' function, like that for other classes with forms.
   * Validation must therefore occur on use, rather than on form submission.
   *
   * @endinternal
   */
  protected function sanitizeSettings() {
    $hideLink = $this->getSetting('hideLink');
    $hideROR = $this->getSetting('hideROR');

    $hideLink = boolval($hideLink);
    $hideROR = boolval($hideROR);

    $this->setSetting('hideLink', $hideLink);
    $this->setSetting('hideROR', $hideROR);
  }

  /**
   * The form is rebuilt on submit.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Set the form to rebuild. The submitted values are maintained in the
    // form state, and used to build the search results in the form definition.
    $form_state->setRebuild(TRUE);
  }

  /**
   * Ajax callback method.
   *
   * It gets executed after getting the results from the
   * autocomplete REST API, accesses the ROR API
   * to get the affiliation information, and updates values
   * in the form elements.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function myAjaxCallback(array &$form, FormStateInterface $form_state) {
    $triggering = $form_state->getTriggeringElement();
    $value = $form_state->getValue($triggering['#parents']);
    $url_encoded_org_name = urlencode($value);

    if (!empty($value)) {
      try {
        $config = $this->configFactory->get('organization_field.settings');
        $ror_api = (string) $config->get('ror_api');
        $response = $this->httpClient->request('GET', $ror_api, [
          'query' => ['affiliation' => $value],
        ]);

        $json_string = (string) $response->getBody();
        $json = json_decode($json_string, TRUE);
        $first = (is_array($json) && isset($json['items'][0])) ? $json['items'][0] : NULL;

        if (!is_array($first) || !isset($first['organization'])) {
          $id = NULL;
          $url_str = NULL;
        }
        else {
          $org = $first['organization'];
          $id = $org['id'] ?? NULL;

          // Choose a single best URL from ROR links with priority:
          // 1) type = website, 2) type = wikipedia, 3) any first non-empty value.
          $links = is_array($org['links'] ?? NULL) ? $org['links'] : [];
          $best_url = NULL;

          // Prefer a website link if available.
          foreach ($links as $link) {
            $type = isset($link['type']) ? strtolower((string) $link['type']) : '';
            if ($type === 'website' && !empty($link['value']) && is_string($link['value'])) {
              $best_url = $link['value'];
              break;
            }
          }

          // Fall back to wikipedia if website not found.
          if ($best_url === NULL) {
            foreach ($links as $link) {
              $type = isset($link['type']) ? strtolower((string) $link['type']) : '';
              if ($type === 'wikipedia' && !empty($link['value']) && is_string($link['value'])) {
                $best_url = $link['value'];
                break;
              }
            }
          }

          // As a last resort, pick the first non-empty value.
          if ($best_url === NULL) {
            foreach ($links as $link) {
              if (!empty($link['value']) && is_string($link['value'])) {
                $best_url = $link['value'];
                break;
              }
            }
          }

          $url_str = $best_url;
        }

        $target_form_element = NestedArray::getValue($form, array_slice($triggering['#array_parents'], 0, -1));

        $target_form_element['organization_name']['#value'] = $value;
        $target_form_element['organization_url']['#attributes']['autocomplete'] = ['off'];
        $target_form_element['organization_url']['#attributes']['role'] = ['presentation'];
        $target_form_element['organization_url']['#value'] = $url_str;
        $target_form_element['organization_ror_id_url']['#value'] = $id;
        $target_form_element['_weight'] = [];
      }
      catch (\Throwable $e) {
        // If the API call fails, keep the form usable and return existing wrapper.
        $target_form_element = NestedArray::getValue($form, array_slice($triggering['#array_parents'], 0, -1));
      }
    }
    else {
      $target_form_element = NestedArray::getValue($form, array_slice($triggering['#array_parents'], 0, -1));
    }

    return $target_form_element;

  }

  /**
   * Defines form elements that represent the organization field widget.
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $index = Html::getUniqueId('osp-org');
    $element['osp_organization_field'] = [
      '#type' => 'details',
      '#title' => $this->t('Organization Details'),
      '#open' => TRUE,
      '#attributes' => [
        'id' => "org-data-wrapper-$index",
      ],
      '#prefix' => '<div id="org-data-wrapper-' . $index . '">',
      '#suffix' => '</div>',
    ];

    $element['osp_organization_field']['organization_name'] = [
      '#title' => $this->t('Organization Name'),
      '#type' => 'textfield',
      '#size' => 60,
      '#maxlength' => 255,
      '#autocomplete_route_name' => 'organization_field.autocomplete',
      '#default_value' => $items[$delta]->organization_name ?? NULL,
      '#description' => $this->t('Example - Your organization'),

      '#attributes' => [
        'autocomplete' => 'off',
        'role' => 'presentation',
      ],

      '#ajax' => [
        'callback' => [$this, 'myAjaxCallback'],
        'event' => 'autocompleteclose',
        'wrapper' => "org-data-wrapper-$index",
        'suppress_required_fields_validation' => TRUE,
      ],

    ];

    // As the string type, Handles multiple URLS
    // by storing it as comma separated values.
    if (!$this->getSetting('hideLink')) {
      $element['osp_organization_field']['organization_url'] = [
        '#title' => $this->t('Organization URL'),
        '#type' => 'textfield',
        '#size' => 60,
        '#default_value' => $items[$delta]->organization_url ?? NULL,
        '#description' => $this->t('Example - http://your organization/'),
        '#maxlength' => 2048,
      ];
    }

    if (!$this->getSetting('hideROR')) {
      $element['osp_organization_field']['organization_ror_id_url'] = [
        '#title' => $this->t('Research Organization Registry (ROR) URL'),
        '#type' => 'textfield',
        '#size' => 60,
        '#maxlength' => 255,
        '#default_value' => $items[$delta]->organization_ror_id_url ?? NULL,
        '#description' => $this->t('Example - https://ror.org/01kg8sb98'),
      ];
    }

    // If cardinality is 1, ensure a proper label is output for the field.
    // Wrap everything in a details element.
    if ($this->fieldDefinition->getFieldStorageDefinition()->getCardinality() == 1) {
      $field_name = $this->fieldDefinition->getName();
      $element += [
        '#type' => 'markup',
        '#markup' => $this->t('<h4 style="margin-bottom: 1em;">@Author</h4><hr style="color: #0a6eb4">',
                ['@Author' => ucfirst($field_name)]),
      ];
    }

    return $element;
  }

  /**
   * Massages the form values into the format expected for field values.
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    foreach ($values as $delta => $value) {
      if (isset($value['osp_organization_field']['organization_name'])) {
        if ($value['osp_organization_field']['organization_name'] === '') {
          $values[$delta]['organization_name'] = NULL;
        }
        else {
          $values[$delta]['organization_name'] = $value['osp_organization_field']['organization_name'];
        }
      }
      else {
        $values[$delta]['organization_name'] = NULL;
      }
      if (isset($value['osp_organization_field']['organization_url'])) {
        if ($value['osp_organization_field']['organization_url'] === '') {
          $values[$delta]['organization_url'] = NULL;
        }
        else {
          $values[$delta]['organization_url'] = $value['osp_organization_field']['organization_url'];
        }
      }
      else {
        $values[$delta]['organization_url'] = NULL;
      }

      if (isset($value['osp_organization_field']['organization_ror_id_url'])) {
        if ($value['osp_organization_field']['organization_ror_id_url'] === '') {
          $values[$delta]['organization_ror_id_url'] = NULL;
        }
        else {
          $values[$delta]['organization_ror_id_url'] = $value['osp_organization_field']['organization_ror_id_url'];
        }
      }
      else {
        $values[$delta]['organization_ror_id_url'] = NULL;
      }
    }

    return $values;
  }

}
