<?php

declare(strict_types=1);

namespace Drupal\orphans_media\Batch;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\media\Entity\Media;
use Drupal\media\MediaInterface;
use Drupal\orphans_media\Event\OrphansMediaDeleteMediaEntityEventInterface;
use Drupal\orphans_media\Event\OrphansMediaPostDeleteMediaEntityEvent;
use Drupal\orphans_media\Event\OrphansMediaPreDeleteMediaEntityEvent;

/**
 * Batch Worker to handle Deleting entity record.
 */
class OrphansMediaDeleteMediaEntityBatchProcess {

  /**
   * Delete Media Batch process.
   *
   * @param array $reference
   *   The media to process.
   *   Where key is media id et value is media name.
   * @param array $context
   *   Batch context.
   *
   * @return void
   *   Return nothing.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public static function delete(array $reference, array &$context): void {
    if (empty($reference)) {
      return;
    }

    $mediaName = array_values($reference)[0];
    $mediaId = array_key_first($reference);

    $context['message'] = \t('Processed Media @name (@id)', [
      '@name' => $mediaName,
      '@id' => $mediaId,
    ]);
    $status = FALSE;
    $media = \Drupal::entityTypeManager()->getStorage('media')->load($mediaId);
    if ($media instanceof Media) {
      try {
        static::dispatchEvent($media, 'pre_delete');
        $mediaDeleted = clone $media;
        $media->delete();
        static::dispatchEvent($mediaDeleted, 'post_delete');
        $status = TRUE;
      }
      catch (EntityStorageException $e) {
        static::getMessenger()->addError($e->getMessage());
      }
    }
    $status ? static::messenger()
      ->addStatus(\t('Successfully deleted Media @name (@id).', [
        '@name' => $mediaName,
        '@id' => $mediaId,
      ])) : static::messenger()
      ->addError(\t('There was a problem deleting the record with ID @mid. Check the logs for more information.', ['@mid' => $mediaId]));
  }

  /**
   * Dispatches an event.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   * @param string $event
   *   The event to dispatch.
   *
   * @return void
   *   Return nothing.
   */
  protected static function dispatchEvent(MediaInterface $media, string $event): void {
    switch ($event) {
      case 'pre_delete':
        \Drupal::service('event_dispatcher')
          ->dispatch(new OrphansMediaPreDeleteMediaEntityEvent($media), OrphansMediaDeleteMediaEntityEventInterface::PRE_DELETE_EVENT);
        break;

      case 'post_delete':
        \Drupal::service('event_dispatcher')
          ->dispatch(new OrphansMediaPostDeleteMediaEntityEvent($media), OrphansMediaDeleteMediaEntityEventInterface::POST_DELETE_EVENT);
        break;
    }
  }

  /**
   * The messenger service.
   */
  protected static function getMessenger(): MessengerInterface {
    // @phpstan-ignore-next-line
    return \Drupal::service('messenger');
  }

  /**
   * Messenger service.
   */
  protected static function messenger(): MessengerInterface {
    return \Drupal::service('messenger');
  }

  /**
   * Finalize batch.
   *
   * @param bool $success
   *   A boolean indicating whether the re-build process has completed.
   * @param array $results
   *   An array of results information.
   * @param array $operations
   *   An array of function calls (not used in this function).
   */
  public static function finishBatch(bool $success, array $results, array $operations): void {
    if (!$success) {
      $error_operation = reset($operations);
      static::getMessenger()
        ->addError(\t('An error occurred while processing @operation with arguments : @args', [
          '@operation' => $error_operation[0],
          '@args' => print_r($error_operation[0], TRUE),
        ]));
    }
  }

  /**
   * Process batch.
   *
   * @param array $references
   *   Array of references where key is media id et value is media name.
   *
   * @return void
   *   Return nothing.
   */
  public static function processBatch(array $references): void {
    if (empty($references)) {
      return;
    }
    $batchDefinition = (new BatchBuilder())
      ->setTitle(\t('Deleting Media entities'))
      ->setErrorMessage(\t('One or more errors were encountered processing the files.'))
      ->setFinishCallback([
        OrphansMediaDeleteMediaEntityBatchProcess::class,
        'finishBatch',
      ])
      ->setProgressMessage(\t('Completed @current of @total operations.'));
    foreach ($references as $mid => $reference) {
      $batchDefinition->addOperation(
        [OrphansMediaDeleteMediaEntityBatchProcess::class, 'delete'],
        [[$mid => $reference]],
      );
    }
    static::batchSet($batchDefinition);
  }

  /**
   * Sets the batch.
   */
  protected static function batchSet(BatchBuilder $batchDefinition): void {
    \batch_set($batchDefinition->toArray());
  }

}
