<?php

namespace Drupal\orphans_media\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;

/**
 * Trait ListFormTrait. List form trait.
 *
 * @package Drupal\orphans_media\Form
 */
trait ListFormTrait {

  /**
   * Build list form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The form structure.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function buildListForm(array &$form, FormStateInterface $form_state): array {
    // Get items per page.
    $itemsPerPage = intval($this->requestStack->getCurrentRequest()->query->get('items_per_page') ?? static::DEFAULT_ITEMS_PER_PAGE);

    // Define table header first (needed for tablesort).
    $tableHeader = [
      'title' => [
        'data' => $this->t('Title'),
        'field' => 'name',
        'specifier' => 'name',
      ],
      'id' => [
        'data' => $this->t('ID'),
        'field' => 'mid',
        'specifier' => 'mid',
      ],
      'type' => [
        'data' => $this->t('Type'),
        'field' => 'bundle',
        'specifier' => 'bundle',
      ],
      'created' => [
        'data' => $this->t('Created'),
        'field' => 'created',
        'specifier' => 'created',
        'sort' => 'desc',
      ],
      'updated' => [
        'data' => $this->t('Updated'),
        'field' => 'changed',
        'specifier' => 'changed',
      ],
      'link' => $this->t('Link'),
      'delete' => $this->t('Delete'),
    ];

    // Get sort parameters from URL.
    $request = $this->requestStack->getCurrentRequest();
    $orderParam = $request->query->get('order');
    $sortParam = $request->query->get('sort');

    // Find the field name from the header based on the order parameter.
    $sortField = 'created';
    $sortOrder = 'DESC';

    if ($orderParam && $sortParam) {
      foreach ($tableHeader as $key => $header) {
        if (is_array($header) && isset($header['data']) && $header['data'] == $orderParam) {
          $sortField = $header['field'];
          $sortOrder = strtoupper($sortParam);
          break;
        }
      }
    }

    // Get filter parameters.
    $filters = $this->getFilterValues();
    $requestedMediaBundles = $filters['media_bundle'];

    // Get media entities.
    $total = $this->orphansMediaManager->getTotalUnusedMedias($requestedMediaBundles, $filters);
    $currentPage = $this->pagerManager->createPager($total, $itemsPerPage)
      ->getCurrentPage();
    $offset = $currentPage * $itemsPerPage;
    $mediaEntities = $this->orphansMediaManager->getUnusedMedias($currentPage, $itemsPerPage, $offset, $requestedMediaBundles, $sortField, $sortOrder, TRUE, $filters);

    // Build table.
    $rows = [];
    foreach ($mediaEntities as $media) {
      $rows[$media->id()] = [
        'title' => Markup::create('<strong>' . $media->label() . '</strong>'),
        'id' => Markup::create('<strong>' . $media->id() . '</strong>'),
        'type' => Markup::create('<strong>' . $media->bundle() . '</strong>'),
        'created' => Markup::create('<strong>' . date('Y-m-d H:i:s', $media->getCreatedTime()) . '</strong>'),
        'updated' => Markup::create('<strong>' . date('Y-m-d H:i:s', $media->getChangedTime()) . '</strong>'),
        'link' => Markup::create('<strong><a class="button" href="' . Url::fromRoute('entity.media.edit_form',
            [
              'media' => $media->id(),
            ])->toString() . '">' . $this->t('Edit') . '</a></strong>'),
        'delete' => Markup::create('<strong><a class="button" href="' . Url::fromRoute('entity.media.delete_form',
            [
              'media' => $media->id(), [['destination' => Url::fromRoute('orphans_media.delete')->toString()]],
            ])->toString() . '">' . $this->t('Delete') . '</a></strong>'),
      ];
    }

    // Build form.
    $form['filter'] = [
      '#type' => 'details',
      '#title' => $this->t('Filters'),
      '#open' => TRUE,
    ];

    // Row 1: Title and all dates.
    $form['filter']['row1'] = [
      '#type' => 'container',
      '#attributes' => ['style' => 'display: flex; gap: 1rem; margin-bottom: 1rem;'],
    ];

    $form['filter']['row1']['title'] = [
      '#title' => $this->t('Title'),
      '#type' => 'textfield',
      '#default_value' => $filters['title'],
      '#placeholder' => $this->t('Search by title...'),
      '#attributes' => ['style' => 'flex: 2;'],
    ];

    $form['filter']['row1']['created_from'] = [
      '#title' => $this->t('Created from'),
      '#type' => 'date',
      '#default_value' => $filters['created_from'],
      '#attributes' => ['style' => 'flex: 1;'],
    ];

    $form['filter']['row1']['created_to'] = [
      '#title' => $this->t('Created to'),
      '#type' => 'date',
      '#default_value' => $filters['created_to'],
      '#attributes' => ['style' => 'flex: 1;'],
    ];

    $form['filter']['row1']['updated_from'] = [
      '#title' => $this->t('Updated from'),
      '#type' => 'date',
      '#default_value' => $filters['updated_from'],
      '#attributes' => ['style' => 'flex: 1;'],
    ];

    $form['filter']['row1']['updated_to'] = [
      '#title' => $this->t('Updated to'),
      '#type' => 'date',
      '#default_value' => $filters['updated_to'],
      '#attributes' => ['style' => 'flex: 1;'],
    ];

    // Row 2: Media bundle, items per page and actions.
    $form['filter']['row2'] = [
      '#type' => 'container',
      '#attributes' => ['style' => 'display: flex; gap: 1rem; align-items: flex-end;'],
    ];

    $form['filter']['row2']['media_bundle'] = [
      '#title' => $this->t('Media bundle'),
      '#type' => 'select',
      '#multiple' => TRUE,
      '#options' => $this->orphansMediaManager->getAvailableMediaBundles(),
      '#default_value' => $requestedMediaBundles,
      '#attributes' => ['style' => 'flex: 1;'],
    ];

    $form['filter']['row2']['items_per_page'] = [
      '#title' => $this->t('Items per page'),
      '#type' => 'select',
      '#options' => [
        25 => 25,
        50 => 50,
        75 => 75,
        100 => 100,
        250 => 250,
        500 => 500,
        750 => 750,
        1000 => 1000,
      ],
      '#default_value' => $itemsPerPage,
      '#attributes' => ['style' => 'flex: 0 0 150px;'],
    ];

    $form['filter']['row2']['spacer'] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => ['style' => 'flex: 1;'],
    ];

    $form['filter']['row2']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Apply filters'),
      '#submit' => [
        [$this, 'submitFilter'],
      ],
      '#attributes' => ['style' => 'margin-right: 0.5rem;'],
    ];

    $form['filter']['row2']['reset'] = [
      '#type' => 'submit',
      '#value' => $this->t('Reset'),
      '#submit' => [
        [$this, 'resetFilter'],
      ],
    ];

    $form['header'] = [
      '#type' => 'html_tag',
      '#tag' => 'h3',
      '#value' => $this->t('Total: @total', ['@total' => $total]),
    ];

    $form['table'] = [
      '#type' => 'tableselect',
      '#header' => $tableHeader,
      '#rows' => $rows,
      '#empty' => $this->t('No unused media found.'),
      '#options' => $rows,
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Delete selected items'),
      '#validate' => [
        [$this, 'validateListForm'],
      ],
      '#disabled' => empty($total),
    ];

    // Pager.
    $form['pager'] = ['#type' => 'pager'];

    return $form;
  }

  /**
   * Validate list form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return void
   *   Nothing.
   */
  public function validateListForm(array &$form, FormStateInterface $form_state): void {
    $medias = array_filter($form_state->getValue('table'));
    if (0 === count($medias)) {
      $form_state->setError($form, $this->t('No items were selected to operate on.'));
    }
  }

  /**
   * Filter submit.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return void
   *   Nothing.
   */
  public function submitFilter(array &$form, FormStateInterface $form_state): void {
    $request = $this->requestStack->getCurrentRequest();
    $request->query->set('items_per_page', $form_state->getValue('items_per_page'));
    $request->query->set('media_bundle', $form_state->getValue('media_bundle'));
    $request->query->set('title', $form_state->getValue('title'));
    $request->query->set('created_from', $form_state->getValue('created_from'));
    $request->query->set('created_to', $form_state->getValue('created_to'));
    $request->query->set('updated_from', $form_state->getValue('updated_from'));
    $request->query->set('updated_to', $form_state->getValue('updated_to'));
  }

  /**
   * Reset filter submit.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return void
   *   Nothing.
   */
  public function resetFilter(array &$form, FormStateInterface $form_state): void {
    $request = $this->requestStack->getCurrentRequest();
    $request->query->remove('items_per_page');
    $request->query->remove('media_bundle');
    $request->query->remove('title');
    $request->query->remove('created_from');
    $request->query->remove('created_to');
    $request->query->remove('updated_from');
    $request->query->remove('updated_to');
  }

  /**
   * Get filter values from request.
   *
   * @return array
   *   The filter values.
   */
  public function getFilterValues(): array {
    $parameters = $this->requestStack->getCurrentRequest()->query->all();
    return [
      'media_bundle' => $parameters['media_bundle'] ?? [],
      'title' => $parameters['title'] ?? '',
      'created_from' => $parameters['created_from'] ?? '',
      'created_to' => $parameters['created_to'] ?? '',
      'updated_from' => $parameters['updated_from'] ?? '',
      'updated_to' => $parameters['updated_to'] ?? '',
    ];
  }

  /**
   * Get default media bundles values.
   *
   * @return array
   *   The default media bundles values.
   */
  public function getRequestedMediaBundlesValues(): array {
    $parameters = $this->requestStack->getCurrentRequest()->query->all();
    return $parameters['media_bundle'] ?? [];
  }

}
