<?php

declare(strict_types=1);

namespace Drupal\orphans_media\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\media\Entity\Media;
use Drupal\orphans_media\Manager\OrphansMediaManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class OrphansMediaDeleteForm.
 *
 *  Form for deleting orphans media.
 *
 * @package Drupal\orphans_media\Form
 */
class OrphansMediaDeleteForm extends ConfirmFormBase {

  use ListFormTrait;
  use ConfirmFormTrait;

  const DEFAULT_ITEMS_PER_PAGE = 25;

  /**
   * Orphans media manager.
   *
   * @var \Drupal\orphans_media\Manager\OrphansMediaManager
   */
  protected OrphansMediaManager $orphansMediaManager;

  /**
   * Pager manager.
   *
   * @var \Drupal\Core\Pager\PagerManagerInterface
   */
  protected PagerManagerInterface $pagerManager;

  /**
   * Create form.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   Container.
   *
   * @return \Drupal\orphans_media\Form\OrphansMediaDeleteForm
   *   The instance.
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->orphansMediaManager = $container->get(OrphansMediaManager::SERVICE_NAME);
    $instance->requestStack = $container->get('request_stack');
    $instance->pagerManager = $container->get('pager.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'orphans_media_delete';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    /** @var array{confirm: true|null, files: string[], op: string} $storage */
    $storage = $form_state->getStorage();
    return !isset($storage['confirm'])
      ? $this->buildListForm($form, $form_state)
      : $this->buildConfirmForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $references = [];
    foreach ($form_state->getValue('table') as $reference) {
      $media = Media::load($reference);
      if ($media) {
        $references[$media->id()] = $media->label();
      }
    }
    $form_state
      ->setStorage([
        'references' => $references,
        'confirm' => TRUE,
      ])
      ->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t("Do you want to delete following record");
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return Url::fromRoute('orphans_media.delete');
  }

}
