<?php

declare(strict_types=1);

namespace Drupal\orphans_media\Manager;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\orphans_media\Batch\OrphansMediaDeleteMediaEntityBatchProcess;

/**
 * Class OrphansMediaManager. Manages orphans media.
 *
 * @package Drupal\orphans_media\Manager
 */
class OrphansMediaManager {

  const SERVICE_NAME = 'orphans_media.manager';

  /**
   * OrphansMediaManager constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   EntityTypeBundleInfo.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   Entity field manager.
   * @param \Drupal\Core\Database\Connection $database
   *   Database.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   Logger.
   */
  public function __construct(protected EntityTypeManagerInterface $entityTypeManager, protected EntityTypeBundleInfoInterface $entityTypeBundleInfo, protected EntityFieldManagerInterface $entityFieldManager, protected Connection $database, protected LoggerChannelInterface $logger) {}

  /**
   * Get all media except used.
   *
   * @param int|null $currentPage
   *   Current page.
   * @param int|null $itemsPerPage
   *   Items per page.
   * @param int|null $offset
   *   Offset.
   * @param array $bundles
   *   Bundles.
   * @param string $sortField
   *   Sort field.
   * @param string $sortOrder
   *   Sort order (ASC or DESC).
   * @param bool $excludeConfigEntityTypes
   *   Exclude configuration entity types.
   * @param array $filters
   *   Additional filters (title, created_from, created_to,
   *   updated_from, updated_to).
   *
   * @return array
   *   Media entities.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getUnusedMedias(?int $currentPage = NULL, ?int $itemsPerPage = NULL, ?int $offset = NULL, array $bundles = [], string $sortField = 'created', string $sortOrder = 'DESC', bool $excludeConfigEntityTypes = TRUE, array $filters = []): array {
    // Get all used media ids.
    $storage = $this->entityTypeManager->getStorage('media');
    $query = $storage->getQuery()
      ->accessCheck()
      ->range($offset, $itemsPerPage)
      ->sort($sortField, strtoupper($sortOrder));

    if (!empty($bundles)) {
      $query->condition('bundle', $bundles, 'IN');
    }

    // Apply filters.
    $this->applyFilters($query, $filters);

    try {
      $usedMediaIds = $this->getUsedMediaIds($excludeConfigEntityTypes);
    }
    catch (\Exception $e) {
      $this->logger->error($e->getMessage());
    }
    if (!empty($usedMediaIds)) {
      $query->condition('mid', $usedMediaIds, 'NOT IN');
    }

    // Set pagination.
    if ($currentPage === NULL || $itemsPerPage === NULL) {
      $query->range($currentPage * $itemsPerPage, $itemsPerPage);
    }
    $mids = $query->execute();
    return !empty($mids) ? $storage->loadMultiple($mids) : [];
  }

  /**
   * Returns an array of media ids referenced by any entity.
   *
   * @param bool $excludeConfigEntityTypes
   *   Exclude configuration entity types.
   *
   * @return array
   *   The media ids.
   *
   * @throws \Exception
   */
  public function getUsedMediaIds(bool $excludeConfigEntityTypes = FALSE): array {
    $allEntityTypes = $this->entityTypeManager->getDefinitions();
    $mids = [];
    foreach ($allEntityTypes as $entityTypes) {
      // Exclude configuration entity types.
      if ($excludeConfigEntityTypes && $entityTypes->getGroup() == 'configuration') {
        continue;
      }
      $mediaIds = $this->fetchReferencedMediaIdsByEntityTypeId($entityTypes->id());
      if ($mediaIds) {
        $mids = array_merge($mids, $mediaIds);
      }
    }
    return array_unique($mids);
  }

  /**
   * Returns an array of media ids referenced by a specific entity type.
   *
   * @param string $entityTypeId
   *   The entity type id.
   *
   * @return array
   *   The media ids.
   *
   * @throws \Exception
   */
  protected function fetchReferencedMediaIdsByEntityTypeId(string $entityTypeId): array {
    $entityFields = $this->getFieldsReferencingMedia($entityTypeId);
    $output = [];
    foreach ($entityFields as $field) {
      $query = $this->database->select($entityTypeId . '__' . $field, 'fd');
      $query->addField('fd', $field . '_target_id');
      $query->addField('fd', $field . '_target_id');
      $res = $query->execute()->fetchAllKeyed(0, 1);
      $output = array_unique(array_merge($output, $res));
    }

    return $output;
  }

  /**
   * Returns an array of media ids referenced by a specific entity type.
   *
   * @param int|string $parentEntityId
   *   The entity type id.
   *
   * @return array
   *   The media ids.
   */
  protected function getFieldsReferencingMedia(int | string $parentEntityId): array {
    $bundles = $this->entityTypeBundleInfo->getBundleInfo($parentEntityId);
    $output = [];
    foreach ($bundles as $bundle => $data) {
      if (!array_key_exists($parentEntityId, $this->entityFieldManager->getFieldMap())) {
        continue;
      }
      $fields = $this->entityFieldManager->getFieldDefinitions($parentEntityId,
        $bundle);
      foreach ($fields as $field_name => $field) {
        $itemDefinition = $field->getItemDefinition();
        if ($itemDefinition?->getSetting('target_type') == 'media') {
          $output[$field_name] = $field_name;
        }
      }
    }
    return $output;
  }

  /**
   * Returns the total number of unused media entities.
   *
   * @param array $bundles
   *   An array of media bundles.
   * @param array $filters
   *   Additional filters (title, created_from, created_to,
   *   updated_from, updated_to).
   * @param bool $excludeConfigEntityTypes
   *   Exclude configuration entity types.
   *
   * @return int
   *   The total number of unused media entities.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getTotalUnusedMedias(array $bundles = [], array $filters = [], bool $excludeConfigEntityTypes = TRUE): int {
    try {
      $usedMediaIds = $this->getUsedMediaIds($excludeConfigEntityTypes);
    }
    catch (\Exception $e) {
      $this->logger->error($e->getMessage());
    }
    $query = $this->entityTypeManager->getStorage('media')
      ->getQuery()
      ->accessCheck();
    if (!empty($usedMediaIds)) {
      $query->condition('mid', $usedMediaIds, 'NOT IN');
    }
    if (!empty($bundles)) {
      $query->condition('bundle', $bundles, 'IN');
    }

    // Apply filters.
    $this->applyFilters($query, $filters);

    return $query->count()->execute();
  }

  /**
   * Deletes media entities.
   *
   * @param array $references
   *   Array of references where key is media id et value is media name.
   *
   * @return void
   *   Return nothing.
   */
  public function deleteMediaBatch(array $references): void {
    (new OrphansMediaDeleteMediaEntityBatchProcess())::processBatch($references);
  }

  /**
   * Returns an array of available media bundles.
   *
   * @return array
   *   An array of available media bundles.
   */
  public function getAvailableMediaBundles(): array {
    $infos = $this->entityTypeBundleInfo->getBundleInfo('media');
    return array_map(function ($info) {
      return $info['label'];
    }, $infos);
  }

  /**
   * Apply filters to the query.
   *
   * @param \Drupal\Core\Entity\Query\QueryInterface $query
   *   The query to apply filters to.
   * @param array $filters
   *   The filters to apply.
   *
   * @return void
   *   Nothing.
   */
  protected function applyFilters($query, array $filters): void {
    // Filter by title.
    if (!empty($filters['title'])) {
      $query->condition('name', '%' . $this->database->escapeLike($filters['title']) . '%', 'LIKE');
    }

    // Filter by created date range.
    if (!empty($filters['created_from'])) {
      $timestamp = strtotime($filters['created_from'] . ' 00:00:00');
      if ($timestamp !== FALSE) {
        $query->condition('created', $timestamp, '>=');
      }
    }
    if (!empty($filters['created_to'])) {
      $timestamp = strtotime($filters['created_to'] . ' 23:59:59');
      if ($timestamp !== FALSE) {
        $query->condition('created', $timestamp, '<=');
      }
    }

    // Filter by updated date range.
    if (!empty($filters['updated_from'])) {
      $timestamp = strtotime($filters['updated_from'] . ' 00:00:00');
      if ($timestamp !== FALSE) {
        $query->condition('changed', $timestamp, '>=');
      }
    }
    if (!empty($filters['updated_to'])) {
      $timestamp = strtotime($filters['updated_to'] . ' 23:59:59');
      if ($timestamp !== FALSE) {
        $query->condition('changed', $timestamp, '<=');
      }
    }
  }

}
