<?php

namespace Drupal\otp_login\Form;

use Drupal\otp_login\Controller\Otp;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class OtpLoginForm.
 */
class OtpLoginForm extends FormBase {

  /**
   * The otp service.
   *
   * @var \Drupal\otp_login\Controller\Otp
   */
  protected $OTP;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
        $container->get('otp_login.OTP')
      );
  }

  /**
   * Creates an object.
   *
   * @param Drupal\otp_login\Controller\Otp $otpLogin
   *   The otp service.
   */
  public function __construct(Otp $otpLogin) {
    $this->OTP = $otpLogin;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'otp_login_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['mobile_number'] = [
      '#type' => 'tel',
      '#title' => $this->t('Mobile Number'),
      '#description' => $this->t('Enter valid mobile number.'),
      '#required' => TRUE,
      '#weight' => '0',
    ];
    $form['otp'] = [
      '#type' => 'fieldset',
      '#prefix' => '<div id="otpWrapper">',
      '#suffix' => '</div>',
    ];
    $form['otp']['generate_otp'] = [
      '#type' => 'submit',
      '#value' => $this->t('Generate OTP'),
      '#weight' => '0',
      '#submit' => ['::otpLoginGenerateOtp'],
      '#ajax' => [
        'callback' => [$this, 'otpLoginGenerateOtpCallback'],
        'wrapper' => 'otpValidateWrapper',
      ],
    ];
    $form['otp']['otp_validate'] = [
      '#type' => 'fieldset',
      '#prefix' => '<div id="otpValidateWrapper">',
      '#suffix' => '</div>',
    ];
    $otp_send = $form_state->get('otp_send');
    if ($otp_send) {
      $form['otp']['otp_validate']['msg'] = [
        '#markup' => $this->t('OTP sent successfully.'),
        '#weight' => '0',
      ];
      $form['otp']['otp_validate']['otp_value'] = [
        '#type' => 'textfield',
        '#title' => $this->t('OTP'),
        '#maxlength' => 6,
        '#size' => 6,
        '#weight' => '1',
      ];
      $form['otp']['otp_validate']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Submit'),
        '#weight' => '2',
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function otpLoginGenerateOtp(array $form, FormStateInterface $form_state) {
    $mobile_number = $form_state->getValue('mobile_number');
    $this->OTP->generateOtp($mobile_number);
    $form_state->set('otp_send', TRUE);
    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function otpLoginGenerateOtpCallback(array &$form, FormStateInterface $form_state) {
    return $form['otp']['otp_validate'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $otp = $form_state->getValue('otp_value');
    $mobile_number = $form_state->getValue('mobile_number');
    $is_invalid_otp = $this->OTP->validateOtp($otp, $mobile_number);
    $user = $this->OTP->otpLoginCheckUserAlreadyExists($mobile_number);

    if ($is_invalid_otp) {
      $this->messenger()->addError($this->t('Incorrect OTP. Please provide correct OTP.'));
      $form_state->setRebuild();
    }
    elseif ($user) {
      $session_id = $this->OTP->userOtpLogin($otp, $mobile_number);
      // Save user cookie.
      user_cookie_save([$mobile_number => $session_id]);
      // Redirect to user profile page.
      $url = Url::fromRoute('user.page');
      $form_state->setRedirectUrl($url);
    }
  }

}
