<?php

namespace Drupal\otp_login;

use Drupal\user\Entity\User;
use Drupal\sms\Message\SmsMessage;
use Drupal\sms\Direction;

/**
 * Class Otp.
 */
class Otp {

  /**
   * {@inheritdoc}
   */
  public function generateOtp($mobile_number) {
    $current_time = \Drupal::time()->getCurrentTime();
    // Generate 6 digit random OTP number.
    $six_digit_random_number = mt_rand(100000, 999999);
    // Send OTP SMS.
    $sms_service = \Drupal::service('sms.provider');
    $sms = (new SmsMessage())
      // Set the message.
      ->setMessage('This is OTP Login: ' . $six_digit_random_number)
      // Set recipient phone number.
      ->addRecipient($mobile_number)
      ->setDirection(Direction::OUTGOING);

    $sms_service->queue($sms);

    $user = $this->otpLoginCheckUserAlreadyExists($mobile_number);
    if (!$user) {
      // Create user object.
      $account = User::create();
      $account->set("name", $mobile_number);
      $account->set("phone_number", $mobile_number);
      $account->save();
      $user = $this->otpLoginCheckUserAlreadyExists($mobile_number);
    }
    $uid = $user->id();

    $userData = \Drupal::service('user.data');
    $data = $userData->get('otp_login', $uid, 'otp_user_data');
    $sessions = $data['sessions'];
    if ($data['otps']) {
      $last_otp_key = end(array_keys($data['otps']));
      $new_otp_key = $last_otp_key + 1;
      $otps = array_merge($data['otps'], [$new_otp_key => ['otp' => $six_digit_random_number, 'otp_time' => $current_time]]);
    }
    else {
      $otps = [['otp' => $six_digit_random_number, 'otp_time' => $current_time]];
    }
    $otp_user_data = [
      "mobile_number" => $mobile_number,
      "otps" => $otps,
      "last_otp_time" => $current_time,
      "sessions" => $sessions,
    ];
    $userData->set('otp_login', $uid, 'otp_user_data', $otp_user_data);
  }

  /**
   * {@inheritdoc}
   */
  public function otpLoginCheckUserAlreadyExists($phone_number) {
    if (empty($phone_number)) {
      return;
    }

    $accounts = \Drupal::entityTypeManager()->getStorage('user')->loadByProperties(['name' => $phone_number]);
    $account = reset($accounts);
    return $account;
  }

  /**
   * {@inheritdoc}
   */
  public function validateOtp($otp, $mobile_number) {
    $users = \Drupal::entityTypeManager()->getStorage('user')
      ->loadByProperties(['name' => $mobile_number]);
    $user = reset($users);
    if ($user) {
      $uid = $user->id();
    }
    // Get OTP from database.
    $userData = \Drupal::service('user.data');
    $data = $userData->get('otp_login', $uid, 'otp_user_data');
    if (!empty($mobile_number) && !empty($otp)) {
      if (array_search($otp, array_column($data['otps'], 'otp')) === FALSE) {
        $otp_incorrect = TRUE;
        return $otp_incorrect;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function userOtpLogin($otp, $mobile_number) {
    $user = $this->otpLoginCheckUserAlreadyExists($mobile_number);
    $uid = $user->id();
    // Generate 6 digit random session id.
    $six_digit_random_sessionid = mt_rand(100000, 999999);

    // Activate user account if not activated.
    if (!$user->isActive()) {
      $user->set("status", 1);
      $user->save();
    }
    $userData = \Drupal::service('user.data');
    $data = $userData->get('otp_login', $uid, 'otp_user_data');
    $lastotptime = $data['last_otp_time'];
    if (($key = array_search($otp, array_column($data['otps'], 'otp'))) !== FALSE) {
      unset($data['otps'][$key]);
      $otps = array_values($data['otps']);
    }
    if ($data['sessions']) {
      $saved_sessions = $data['sessions'];
      $last_session_key = end(array_keys($saved_sessions));
      $new_session_key = $last_session_key + 1;
      $six_digit_sessionid_with_key = [$new_session_key => $six_digit_random_sessionid];
      $sessions = array_merge($saved_sessions, $six_digit_sessionid_with_key);
    }
    else {
      $sessions = [$six_digit_random_sessionid];
    }
    $session_user_data = [
      "mobile_number" => $mobile_number,
      "otps" => $otps,
      "last_otp_time" => $lastotptime,
      "sessions" => $sessions,
    ];
    $userData->set('otp_login', $uid, 'otp_user_data', $session_user_data);

    // Login programmatically as a user.
    $user = User::load($uid);
    user_login_finalize($user);

    return $six_digit_random_sessionid;
  }

  /**
   * {@inheritdoc}
   */
  public function validateSessionid($session_id, $mobile_number) {
    $users = \Drupal::entityTypeManager()->getStorage('user')
      ->loadByProperties(['name' => $mobile_number]);
    $user = reset($users);
    if ($user) {
      $uid = $user->id();
    }
    // Get sessions from database.
    $userData = \Drupal::service('user.data');
    $data = $userData->get('otp_login', $uid, 'otp_user_data');

    if (!empty($mobile_number) && !empty($session_id)) {
      if (!in_array($session_id, $data['sessions'])) {
        $session_id_incorrect = TRUE;
        return $session_id_incorrect;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function userOtpLogout($session_id, $mobile_number) {
    $users = \Drupal::entityTypeManager()->getStorage('user')
      ->loadByProperties(['name' => $mobile_number]);
    $user = reset($users);
    if ($user) {
      $uid = $user->id();
    }
    $userData = \Drupal::service('user.data');
    $data = $userData->get('otp_login', $uid, 'otp_user_data');
    $lastotptime = $data['last_otp_time'];
    if (($key = array_search($session_id, $data['sessions'])) !== FALSE) {
      unset($data['sessions'][$key]);
      $sessions = array_values($data['sessions']);
    }
    else {
      $sessions = $data['sessions'];
    }
    $session_user_data = [
      "mobile_number" => $mobile_number,
      "otps" => $data['otps'],
      "last_otp_time" => $lastotptime,
      "sessions" => $sessions,
    ];
    $userData->set('otp_login', $uid, 'otp_user_data', $session_user_data);
  }

}
