<?php

declare(strict_types=1);

namespace Drupal\Tests\otsdk\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\WaitTerminateTestTrait;
use Drupal\otsdk\Configuration\Resolver\SettingsResolver;
use Drupal\otsdk\OtSdkAutoloadMiddleware;
use OpenTelemetry\SDK\Common\Configuration\Variables;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\Process\PhpExecutableFinder;
use Symfony\Component\Process\Process;

/**
 * End-to-end test for OpenTelemetry SDK autoloading and configuration.
 *
 * * Spawns a an instance of the php built-in server with a script which dumps
 *   all data sent to it.
 * * The php built-in server chooses a random free port to listen to. This port
 *   is captured.
 * * The simpletest child site is configured using otsdk autoloading.
 * * Logs and metrics are discarded, traces are recorded and sent to using the
 *   http/json format of the otlp protocol to the dump server script.
 * * The test runner fetches the front page.
 * * A trace from this request is collected by OpenTelemetry and sent to the
 *   dump server.
 * * The test runner waits for the dump file to appear and checks its content
 *   structure.
 */
#[Group('otsdk')]
#[CoversClass(OtSdkAutoloadMiddleware::class)]
#[CoversClass(SettingsResolver::class)]
final class OtSdkTest extends BrowserTestBase {

  use WaitTerminateTestTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['otsdk'];

  /**
   * Path to the PHP binary.
   */
  protected string $php;

  /**
   * The dump server process.
   */
  protected Process $server;

  /**
   * The dump server port.
   */
  protected int $port = 0;

  /**
   * The working directory.
   */
  protected string $workingDirectory;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->php = (new PhpExecutableFinder())->find();
    $this->assertIsString($this->php);

    $this->workingDirectory = $this->siteDirectory . '/files/otsdk-test';
    mkdir($this->workingDirectory, recursive: TRUE);
    file_put_contents(
      $this->workingDirectory . '/dump.php',
      '<?php copy("php://input", "body.json");'
    );

    $this->server = new Process(
      [$this->php, '-S', '127.0.0.1:0', 'dump.php'],
      cwd: $this->workingDirectory,
    );
    $this->server->start();

    $port = 0;
    $this->server->waitUntil(function ($type, $output) use (&$port): bool {
      if ($type === Process::ERR && preg_match('|http://127.0.0.1:(\d+)|', $output, $matches) !== FALSE) {
        $port = $matches[1];
        return TRUE;
      }
      else {
        return FALSE;
      }
    });

    $this->port = intval($port);
    $this->assertGreaterThan(0, $this->port);
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    if ($this->server->isRunning()) {
      $this->server->stop();
    }

    parent::tearDown();
  }

  /**
   * Test callback.
   */
  public function testOtSdkOtlpExporter(): void {
    // Set up OpenTelemetry SDK autoloading with active traces.
    $settings['settings']['otsdk'][Variables::OTEL_PHP_AUTOLOAD_ENABLED] = (object) [
      'value' => TRUE,
      'required' => TRUE,
    ];
    $settings['settings']['otsdk'][Variables::OTEL_PHP_EXPERIMENTAL_AUTO_ROOT_SPAN] = (object) [
      'value' => TRUE,
      'required' => TRUE,
    ];
    $settings['settings']['otsdk'][Variables::OTEL_LOGS_EXPORTER] = (object) [
      'value' => 'none',
      'required' => TRUE,
    ];
    $settings['settings']['otsdk'][Variables::OTEL_METRICS_EXPORTER] = (object) [
      'value' => 'none',
      'required' => TRUE,
    ];
    $settings['settings']['otsdk'][Variables::OTEL_TRACES_EXPORTER] = (object) [
      'value' => 'otlp',
      'required' => TRUE,
    ];
    $settings['settings']['otsdk'][Variables::OTEL_EXPORTER_OTLP_ENDPOINT] = (object) [
      'value' => "http://127.0.0.1:{$this->port}/",
      'required' => TRUE,
    ];
    $settings['settings']['otsdk'][Variables::OTEL_EXPORTER_OTLP_PROTOCOL] = (object) [
      'value' => 'http/json',
      'required' => TRUE,
    ];
    $this->writeSettings($settings);

    $this->setWaitForTerminate();
    $this->drupalGet('<front>');

    $resultFile = $this->workingDirectory . '/body.json';
    for ($attempt = 0; $attempt < 5; $attempt++) {
      if (file_exists($resultFile)) {
        break;
      }
      usleep(1000 * (2 ** $attempt));
    }

    $this->assertFileExists($resultFile);
    $body = file_get_contents($resultFile);
    $json = json_decode($body, associative: TRUE, flags: JSON_THROW_ON_ERROR);

    $this->assertIsArray($json);
    $this->assertIsArray($json['resourceSpans']);
  }

}
