<?php

namespace Drupal\override_cache_control_headers\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Url;
use Drupal\override_cache_control_headers\Utility;
use Drupal\override_cache_control_headers\UtilityInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a form that configures forms module settings.
 */
class OverrideCacheControlHeadersSettingsForm extends ConfigFormBase implements UtilityInterface {

  /**
   * Utility object.
   *
   * @var \Drupal\override_cache_control_headers\Utility
   */
  protected $utility;

  /**
   * The Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Initiate utility and messenger service.
   *
   * @param Drupal\override_cache_control_headers\Utility $utility
   *   Cache control header utility class.
   * @param Drupal\Core\Messenger\MessengerInterface $messenger
   *   Messanger service Interface.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    Utility $utility,
    MessengerInterface $messenger,
    StateInterface $state,
  ) {
    $this->utility = $utility;
    $this->messenger = $messenger;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('override_cache_control_headers_utility'),
      $container->get('messenger'),
      $container->get('state'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'override_cache_control_headers_configuration';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      static::SETTINGS,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $help_url = Url::fromUserInput('/admin/help/override_cache_control_headers');
    $help_link = Link::fromTextAndUrl($this->t('click here'), $help_url);
    $config = $this->config(static::SETTINGS);
    $form['urls_header'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Please enter the URL and headers separated by "|" symbol.'),
      '#default_value' => $config->get('urls_header') ?? '',
      '#description' => $this->t(
        'Example: 
        <b>@example</b>
        <br>Enter one entry per line.
        <br>For a list of possible values, @link.',
        [
          '@example' => '/sitemap.xml|must-revalidate, no-cache, private',
          '@link' => $help_link->toString(),
        ]
      ),
    ];
    $form['urls_header_temp_outer'] = [
      '#type' => 'details',
      '#title' => $this->t('Set the URL and headers for temporary time period'),
      '#open' => TRUE,
    ];
    $form['urls_header_temp_outer']['urls_header_temp'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Please enter the URL, headers and expiry time in minutes separated by "|" symbol.'),
      '#default_value' => $config->get('urls_header_temp') ?? '',
      '#description' => $this->t(
        'Example: 
        <b>@example</b>
        <br>Enter one entry per line.
        <br>The expiration time is calculated in number of minutes.
        <br>For a list of possible values, @link.',
        [
          '@example' => '/sitemap.xml|must-revalidate, no-cache, private|10',
          '@link' => $help_link->toString(),
        ]
      ),
    ];
    $urls_header_tmp = $this->utility->getTempHeadersInState();
    if (!empty($urls_header_tmp)) {
      $field_output = [];
      foreach ($urls_header_tmp as $header) {
        $exploded = explode('|', $header);
        if (isset($exploded[4])) {
          $exploded[2] = $this->t(
            'Expires in :- @date',
            [
              '@date' => date('Y-m-d H:i:s', $exploded[4]),
            ]
          );
          $field_output[] = implode('|', array_slice($exploded, 0, 3));
        }
      }
      $form['urls_header_temp_outer']['urls_header_tmp'] = [
        '#type' => 'textarea',
        '#title' => $this->t('URL headers configured for temporary time period.'),
        '#default_value' => (implode(PHP_EOL, $field_output)),
        '#disabled' => TRUE,
      ];
      $form['urls_header_temp_outer']['delete_urls_header_tmp'] = [
        '#type' => 'submit',
        '#name' => 'delete_urls_header_tmp',
        '#value' => $this->t("Delete All Entry"),
        '#submit' => [[$this, 'deleteAllUrlsHeaderTemp']],
        '#attributes' => [
          'onclick' => 'if(!confirm(Drupal.t("Do you really want to proceed, this will delete all entries?"))){return false;}',
        ],
        '#limit_validation_errors' => [],
      ];
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAllUrlsHeaderTemp(array &$form, FormStateInterface $form_state): void {
    $this->utility->deleteAllTempurls();
    $this->messenger->addMessage($this->t('All entries are successfully deleted.'));
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $fields = [
      'urls_header',
      'urls_header_temp',
    ];
    $validation_passed = TRUE;
    foreach ($fields as $field) {
      if (!empty($values[$field])) {
        $urls_headers = array_map('trim', explode(PHP_EOL, $values[$field]));
        if ((is_array($urls_headers)) && (!empty($urls_headers))) {
          $response = $this->utility->validateCacheControlStrings($urls_headers, $field);
          if (!empty($response)) {
            $validation_passed = FALSE;
            $form_state->setErrorByName($field);
            foreach ($response as $error) {
              $this->messenger->addError($error);
            }
          }
        }
      }
    }
    if ($validation_passed) {
      $this->checkDuplicateUrls($values, $form_state);
    }
  }

  /**
   * Checks for duplicate URLs in the provided values.
   *
   * @param array $values
   *   The form values to check for duplicates.
   * @param Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   */
  private function checkDuplicateUrls(array $values, FormStateInterface $form_state): void {
    $fields = [
      'urls_header',
      'urls_header_temp',
    ];
    $urls_headers = [];
    foreach ($fields as $field) {
      if (!empty($values[$field])) {
        $urls_headers[$field] = array_map('trim', explode(PHP_EOL, $values[$field]));
      }
    }
    $responses = $this->utility->validateDuplicateUrls($urls_headers);
    foreach ($responses as $field_name => $response) {
      $form_state->setErrorByName($field_name);
      foreach ($response as $message) {
        $this->messenger->addError($message);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    $url_invalidate_altered = FALSE;
    $urls_added = [];
    // Process 'urls_header' field.
    $url_invalidate_altered = $this->processUrlsHeader($values['urls_header'], $urls_added);
    // Process 'urls_header_temp' field.
    if (!empty($values['urls_header_temp'])) {
      $this->utility->setTempHeadersInState(explode(PHP_EOL, $values['urls_header_temp']));
      $url_invalidate_altered = TRUE;
    }
    // Invalidate cache and trigger hooks if necessary.
    if ($url_invalidate_altered) {
      $this->utility->invalidateCache();
      if (!empty($urls_added)) {
        $this->triggerUrlAddedHooks($urls_added);
      }
    }
    parent::submitForm($form, $form_state);
  }

  /**
   * Processes the 'urls_header' field and updates configuration.
   *
   * @param string $urls_header
   *   The URLs and headers entered in the form.
   * @param array &$urls_added
   *   An array to store newly added URLs.
   *
   * @return bool
   *   TRUE if the configuration was altered, FALSE otherwise.
   */
  private function processUrlsHeader(string $urls_header, array &$urls_added): bool {
    $url_invalidate_altered = TRUE;
    $existing_urls_header = $this->configFactory->get(static::SETTINGS)->get('urls_header');
    $urls_header_array = array_map('trim', explode(PHP_EOL, $urls_header));
    $existing_urls_header_array = array_map('trim', explode(PHP_EOL, $existing_urls_header));
    $urls_removed = array_filter(array_diff($existing_urls_header_array, $urls_header_array));
    $urls_added = array_filter(array_diff($urls_header_array, $existing_urls_header_array));
    if (empty($urls_removed) && empty($urls_added)) {
      $url_invalidate_altered = FALSE;
    }
    $this->configFactory->getEditable(static::SETTINGS)
      ->set('urls_header', $urls_header)
      ->save();
    return $url_invalidate_altered;
  }

  /**
   * Triggers hooks for newly added URLs.
   *
   * @param array $urls_added
   *   An array of newly added URLs.
   */
  private function triggerUrlAddedHooks(array $urls_added): void {
    $urls = [];
    foreach ($urls_added as $url_added) {
      $urls[] = explode('|', $url_added)[0];
    }
    $this->utility->triggerHook($urls);
  }

}
