<?php

namespace Drupal\override_cache_control_headers;

use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManager;
use Drupal\Core\Path\PathValidator;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\path_alias\AliasManager;

/**
 * Class to handle the helper functions.
 */
class Utility implements UtilityInterface {
  use StringTranslationTrait;

  /**
   * Path validator object.
   *
   * @var \Drupal\Core\Path\PathValidator
   */
  protected $pathValidator;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The cache tags invalidator.
   *
   * @var \Drupal\Core\Cache\CacheTagsInvalidatorInterface
   */
  protected $cacheTagsInvalidator;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The path alias.
   *
   * @var \Drupal\path_alias\AliasManager
   */
  protected $pathAliasManager;

  /**
   * The language Manager.
   *
   * @var Drupal\Core\Language\LanguageManager
   */
  protected $languageManager;

  /**
   * The module handler.
   *
   * @var Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Utility class contructor.
   *
   * @param \Drupal\Core\Path\PathValidator $pathValidator
   *   Path validator object.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Cache\CacheTagsInvalidatorInterface $cacheTagsInvalidator
   *   The cache invalidator service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\path_alias\AliasManager $pathAliasManager
   *   The path alias manager manager.
   * @param Drupal\Core\Language\LanguageManager $languageManager
   *   The language manager.
   * @param Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(
    PathValidator $pathValidator,
    StateInterface $state,
    CacheTagsInvalidatorInterface $cacheTagsInvalidator,
    ConfigFactoryInterface $configFactory,
    AliasManager $pathAliasManager,
    LanguageManager $languageManager,
    ModuleHandlerInterface $module_handler,
  ) {
    $this->pathValidator = $pathValidator;
    $this->state = $state;
    $this->cacheTagsInvalidator = $cacheTagsInvalidator;
    $this->configFactory = $configFactory;
    $this->pathAliasManager = $pathAliasManager;
    $this->languageManager = $languageManager;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Validate the cache control headers and URLs.
   *
   * @param array $urls_headers
   *   List of URLs and cache control headers.
   * @param string $pattern_type
   *   Regex pattern type.
   */
  public function validateCacheControlStrings(
    array $urls_headers,
    string $pattern_type = '',
  ): array {
    // Regular expression pattern for validation.
    switch ($pattern_type) {
      case 'urls_header_temp':
        $pattern = '#^(/[a-zA-Z0-9_\-/.]*\*?[a-zA-Z0-9_\-/.]*(?:\?[a-zA-Z0-9_\-=&%]*)?)\|([a-zA-Z0-9\- ,=]+)\|([0-9]+)$#';
        break;

      default:
        $pattern = '#^(/[a-zA-Z0-9_\-/.]*\*?[a-zA-Z0-9_\-/.]*(?:\?[a-zA-Z0-9_\-=&%]*)?)\|([a-zA-Z0-9\- ,=]+)$#';
        break;
    }
    $response = [];
    $urls_headers = array_filter($urls_headers);
    $pattern_wildcard = '#^(/[a-zA-Z0-9_\-/.]*\*+[a-zA-Z0-9_\-/.]*)$#';
    foreach ($urls_headers as $urls_header) {
      if (preg_match($pattern, $urls_header, $matches)) {
        // Extracted URL.
        $url = $matches[1];
        // Extracted cache-control header.
        $cacheControl = $matches[2];
        // Validate the URL is valid internal URL.
        if (!preg_match($pattern_wildcard, $url) && !$this->pathValidator->isValid($url)) {
          // Validate with alias list, possible to match for any language.
          $languages = array_keys($this->languageManager->getLanguages());
          $url_match_found = FALSE;
          foreach ($languages as $language) {
            $result = $this->pathAliasManager->getPathByAlias($url, $language);
            if ($result != $url) {
              $url_match_found = TRUE;
              break;
            }
          }
          if (!$url_match_found) {
            $response[] = $this->t('This @url is not a valid internal URL.', ['@url' => $url]);
          }
        }
        // List of valid Cache-Control directives.
        $validDirectives = [
          'must-revalidate',
          'no-cache',
          'no-store',
          'public',
          'private',
          'proxy-revalidate',
          'max-age',
          's-maxage',
        ];
        // Split cache-control string and validate each directive.
        $directives = array_map('trim', explode(',', $cacheControl));
        if ((is_array($directives)) && (!empty($directives))) {
          foreach ($directives as $directive) {
            $directiveParts = explode('=', $directive);
            if (!in_array($directiveParts[0], $validDirectives)) {
              $response[] = $this->t(
                'This @directive cache-control directive is invalid.',
                ['@directive' => $directiveParts[0]]
              );
            }
          }
        }
      }
      else {
        $response[] = $this->t('This @urls_header format is invalid.', ['@urls_header' => $urls_header]);
      }
    }
    return $response;
  }

  /**
   * Get the values from state entry.
   */
  public function getTempHeadersInState(): array {
    return $this->state->get(self::STATE_NAME, []);
  }

  /**
   * Set temporary headers in the state system.
   *
   * @param array $headers
   *   An array of headers to be stored temporarily.
   */
  public function setTempHeadersInState(array $headers): void {
    $urls_headers = $this->getTempHeadersInState();
    $urls = [];
    foreach ($headers as $header) {
      $exploded = explode('|', $header);
      $start_time = strtotime('now');
      $exploded[] = $start_time;
      $exploded[] = $start_time + ($exploded[2] * 60);
      $urls_headers[] = implode('|', $exploded);
      $urls[] = $exploded[0];
    }
    $this->state->set(self::STATE_NAME, $urls_headers);
    $this->triggerHook($urls);
  }

  /**
   * Reset temporary headers in the state system.
   *
   * @param array $headers
   *   An array of headers to be stored temporarily.
   */
  public function resetTempHeadersInState(array $headers): void {
    $this->state->set(self::STATE_NAME, $headers);
    $this->invalidateCache();
  }

  /**
   * Delete all entry in state table entry.
   */
  public function deleteAllTempurls() {
    $this->state->set(self::STATE_NAME, []);
    $this->invalidateCache();
  }

  /**
   * Processes temporary URL headers.
   *
   * @param array $urls_header
   *   Url header loaded from states value.
   * @param bool $state_changed
   *   Indicate whether any chnage in state entry.
   */
  public function processTempUrlHeaders(array $urls_header, bool &$state_changed): array {
    $temp = $urls_header;
    $now = strtotime('now');
    foreach ($temp as $key => $header) {
      $exploded = explode('|', $header);
      if ($now > $exploded[4]) {
        unset($urls_header[$key]);
        $state_changed = TRUE;
      }
    }
    return array_values($urls_header);
  }

  /**
   * Invalidate config cache tags.
   */
  public function invalidateCache(): void {
    $this->configFactory->reset(self::SETTINGS);
    $this->cacheTagsInvalidator->invalidateTags(['config:' . self::SETTINGS]);
  }

  /**
   * Trigger a custom hook for cache control headers.
   *
   * @param array $urls
   *   The list of URLs to be overwritten.
   */
  public function triggerHook(array $urls): void {
    try {
      $this->moduleHandler->invokeAll('override_cache_control_headers', [$urls]);
    }
    catch (\Exception $e) {
    }
  }

  /**
   * Validate if URL is duplicated in config.
   *
   * @param array $urls_headers
   *   Array contains URLs and headers.
   * @param bool $validate_config_data
   *   Boolean value indicating whether the current validation needs to done
   *     against the URLs stored in configuration also.
   */
  public function validateDuplicateUrls(array $urls_headers, bool $validate_config_data = FALSE): array {
    $urls_header_tmp = $this->getTempHeadersInState();
    $all_urls = $response = [];
    foreach ($urls_header_tmp as $url_header_tmp) {
      $all_urls[explode('|', $url_header_tmp)[0]] = TRUE;
    }
    if ($validate_config_data) {
      $urls_header = $this->configFactory->getEditable(self::SETTINGS)->get('urls_header');
      $urls_header = array_map('trim', explode(PHP_EOL, $urls_header));
      foreach ($urls_header as $url_header) {
        $all_urls[explode('|', $url_header)[0]] = TRUE;
      }
    }
    foreach ($urls_headers as $field_name => $urls_header) {
      if (!is_array($urls_header)) {
        $urls_header = [$urls_header];
      }
      foreach ($urls_header as $entry) {
        $url = explode('|', $entry)[0] ?? '';
        if (!empty($url)) {
          if (isset($all_urls[$url])) {
            $response[$field_name][] = $this->t('Duplicate URL found: @url.', [
              '@url' => $url,
            ]);
          }
          $all_urls[$url] = TRUE;
        }
      }
    }
    return $response;
  }

}
