<?php

namespace Drupal\Tests\override_cache_control_headers\Unit;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Path\PathMatcherInterface;
use Drupal\Core\State\StateInterface;
use Drupal\override_cache_control_headers\Utility;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\override_cache_control_headers\EventSubscriber\OverrideCacheControlHeaders;
use PHPUnit\Framework\TestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\HttpKernelInterface;

/**
 * Unit test for the OverrideCacheControlHeaders class.
 *
 * @group override_cache_control_headers
 */
class OverrideCacheControlHeadersTest extends TestCase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;
  
  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The utility service.
   *
   * @var \Drupal\override_cache_control_headers\Utility
   */
  protected $utility;

  /**
   * The path matcher service.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface
   */
  protected $pathMatcher;

  /**
   * The event subscriber.
   *
   * @var \Drupal\override_cache_control_headers\EventSubscriber\OverrideCacheControlHeaders
   */
  protected $eventSubscriber;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Mock the config factory.
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->state = $this->createMock(StateInterface::class);
    $this->utility = $this->createMock(Utility::class);
    $this->pathMatcher = $this->createMock(PathMatcherInterface::class);

    // Create the event subscriber.
    $this->eventSubscriber = new OverrideCacheControlHeaders(
      $this->configFactory,
      $this->state,
      $this->utility,
      $this->pathMatcher,
    );
  }

  /**
   * Tests the onViewResponse method.
   */
  public function testOnViewResponse() {
    // Mock the configuration.
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->once())
      ->method('get')
      ->with('urls_header')
      ->willReturn("/example|no-cache\n/another|private");

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('override_cache_control_headers.settings')
      ->willReturn($config);

    // Mock the utility methods.
    $this->utility->expects($this->once())
      ->method('getTempHeadersInState')
      ->willReturn(['/temp|no-store']);

    $this->utility->expects($this->once())
      ->method('processTempUrlHeaders')
      ->with(['/temp|no-store'], $this->anything())
      ->willReturn(['/temp|no-store']);

    $this->utility->expects($this->any())
      ->method('resetTempHeadersInState');

    // Create a request and response.
    $request = new Request([], [], [], [], [], ['REQUEST_URI' => '/example']);
    $response = new Response();

    // Create the response event.
    $event = new ResponseEvent(
      $this->createMock(HttpKernelInterface::class),
      $request,
      HttpKernelInterface::MASTER_REQUEST,
      $response
    );

    // Call the onViewResponse method.
    $this->eventSubscriber->onViewResponse($event);

    // Assert that the Cache-Control header is set correctly.
    $this->assertStringContainsString('no-cache', $response->headers->get('Cache-Control'));
  }

  /**
   * Tests the setHeaders method.
   */
  public function testSetHeaders() {
    // Create a request and response.
    $request = new Request([], [], [], [], [], ['REQUEST_URI' => '/example']);
    $response = new Response();

    // Create the response event.
    $event = new ResponseEvent(
      $this->createMock(HttpKernelInterface::class),
      $request,
      HttpKernelInterface::MASTER_REQUEST,
      $response
    );

    // Call the private setHeaders method using reflection.
    $reflection = new \ReflectionClass($this->eventSubscriber);
    $method = $reflection->getMethod('setHeaders');
    $method->setAccessible(true);

    // Test with valid headers.
    $urls_headers = ['/example|no-cache'];
    $method->invokeArgs($this->eventSubscriber, [$urls_headers, $event]);

    // Assert that the Cache-Control header is set correctly.
    $this->assertStringContainsString('no-cache', $response->headers->get('Cache-Control'));
  }

  /**
   * Tests the getSubscribedEvents method.
   */
  public function testGetSubscribedEvents() {
    $events = OverrideCacheControlHeaders::getSubscribedEvents();
    $this->assertArrayHasKey('kernel.response', $events);
    $this->assertEquals(['onViewResponse', 10000], $events['kernel.response'][0]);
  }

}