<?php

namespace Drupal\override_cache_control_headers\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Path\PathMatcherInterface;
use Drupal\Core\State\StateInterface;
use Drupal\override_cache_control_headers\Utility;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Set the configured cache control headers against the provided URL.
 */
class OverrideCacheControlHeaders implements EventSubscriberInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The utility service.
   *
   * @var \Drupal\override_cache_control_headers\Utility
   */
  protected $utility;

  /**
   * The path matcher service.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface
   */
  protected $pathMatcher;

  /**
   * Override cache control header constructor.
   *
   * Initialize the config factory object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\override_cache_control_headers\Utility $utility
   *   The utility service.
   * @param \Drupal\Core\Path\PathMatcherInterface $path_matcher
   *   The path matcher service.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    StateInterface $state,
    Utility $utility,
    PathMatcherInterface $path_matcher,
  ) {
    $this->configFactory = $configFactory;
    $this->state = $state;
    $this->utility = $utility;
    $this->pathMatcher = $path_matcher;
  }

  /**
   * Modifies the response to disable cache for a specific route.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event that contains the request and response objects.
   */
  public function onViewResponse(ResponseEvent $event) {
    // Check if the request path matches the specific route.
    $urls_header = $this->configFactory->get('override_cache_control_headers.settings')->get('urls_header');
    $set_headers = [];
    if (!empty($urls_header)) {
      $urls_header = trim($urls_header);
      $urls_headers = explode(PHP_EOL, $urls_header);
      if ((is_array($urls_headers)) && (!empty($urls_headers))) {
        $set_headers = array_filter($urls_headers);
      }
    }
    $urls_header = $this->utility->getTempHeadersInState();
    if (!empty($urls_header)) {
      $state_changed = FALSE;
      $urls_header = $this->utility->processTempUrlHeaders($urls_header, $state_changed);
      if ($state_changed) {
        $this->utility->resetTempHeadersInState($urls_header);
      }
      $set_headers = array_merge($set_headers, $urls_header);
    }
    if (!empty($set_headers)) {
      $this->setHeaders($set_headers, $event);
    }
  }

  /**
   * Sets the headers for the given URLs.
   *
   * @param array $urls_headers
   *   An array of URLs for which the headers need to be set.
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event that contains the request and response objects.
   */
  private function setHeaders(array $urls_headers, ResponseEvent $event) {
    // Get the current request.
    $request = $event->getRequest();
    // Get the base path to support subdirectory installations.
    $base_path = $request->getBasePath();
    // Get the current URI path.
    $current_uri = $request->getPathInfo();
    // Loop through each URL header configuration.
    foreach ($urls_headers as $urls_header) {
      $exploded = explode('|', $urls_header);
      $url_to_match = $exploded[0];
      // Check if the configured URL has query parameters.
      if (strpos($url_to_match, '?') !== FALSE) {
        // Use full request URI with query parameters.
        $current_uri = $request->getRequestUri();
      }
      if ($this->pathMatcher->matchPath($current_uri, $base_path . $url_to_match)) {
        // Explode the header to get the cache control value.
        $response = $event->getResponse();
        // Set Cache-Control header to disable.
        // caching for this specific route.
        $response->headers->set('Cache-Control', $exploded[1]);
        return;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = [];
    $events[KernelEvents::RESPONSE][] = ['onViewResponse', 10000];
    return $events;
  }

}
