<?php

namespace Drupal\Tests\override_cache_control_headers\Unit;

use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManager;
use Drupal\Core\Path\PathValidator;
use Drupal\Core\State\StateInterface;
use Drupal\override_cache_control_headers\Utility;
use Drupal\Tests\UnitTestCase;
use Drupal\path_alias\AliasManager;

/**
 * Unit test for the Utility class.
 *
 * @group override_cache_control_headers
 */
class UtilityTest extends UnitTestCase {

  /**
   * The Utility class instance.
   *
   * @var \Drupal\override_cache_control_headers\Utility
   */
  protected $utility;

  /**
   * Pathvalidator
   * 
   * @var \Drupal\Core\Path\PathValidator
   */
  protected $pathValidator;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The cache tags invalidator.
   *
   * @var \Drupal\Core\Cache\CacheTagsInvalidatorInterface
   */
  protected $cacheTagsInvalidator;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The path alias.
   *
   * @var \Drupal\path_alias\AliasManager
   */
  protected $pathAliasManager;

  /**
   * The language Manager.
   *
   * @var Drupal\Core\Language\LanguageManager
   */
  protected $languageManager;

  /**
   * The module handler.
   *
   * @var Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Mock the PathValidator service.
    $this->pathValidator = $this->getMockBuilder(PathValidator::class)
                            ->disableOriginalConstructor()
                            ->setMethods(['isValid'])
                            ->getMock();
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->state = $this->createMock(StateInterface::class);
    $this->cacheTagsInvalidator = $this->createMock(CacheTagsInvalidatorInterface::class);
    $this->pathAliasManager = $this->createMock(AliasManager::class);
    $this->languageManager = $this->createMock(LanguageManager::class);
    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);

    // Mock the StringTranslationTrait.
    $this->utility = $this->getMockBuilder(Utility::class)
      ->setConstructorArgs([
        $this->pathValidator,
        $this->state,
        $this->cacheTagsInvalidator,
        $this->configFactory,
        $this->pathAliasManager,
        $this->languageManager,
        $this->moduleHandler,
      ])
      ->onlyMethods(['t'])
      ->getMock();
    $this->languageManager->expects($this->any())->method('getLanguages')->willReturn(['en']);
    // Mock the translation method.
    $this->utility->method('t')
      ->willReturnCallback(function ($string, array $args = []) {
        return strtr($string, $args);
      });
  }

  /**
   * Tests validateCacheControlStrings with valid input.
   */
  public function testValidateCacheControlStringsValid() {
    $urls_headers = [
      '/valid-path|public, max-age=3600',
      '/another-valid-path|private, no-cache',
    ];
    $this->pathValidator->expects($this->any())->method('isValid')->willReturn(true);
    $result = $this->utility->validateCacheControlStrings($urls_headers);
    $this->assertEmpty($result, 'No errors should be returned for valid input.');
  }

  /**
   * Tests validateCacheControlStrings with invalid URL.
   */
  public function testValidateCacheControlStringsInvalidUrl() {
    $urls_headers = [
      'invalid-path|public, max-age=3600',
    ];
    $this->pathValidator->expects($this->any())->method('isValid')->willReturn(false);
    $result = $this->utility->validateCacheControlStrings($urls_headers);
    $this->assertNotEmpty($result, 'Errors should be returned for invalid URL.');
    $this->assertStringContainsString('This invalid-path|public, max-age=3600 format is invalid.', $result[0]);
  }

  /**
   * Tests validateCacheControlStrings with invalid cache-control directive.
   */
  public function testValidateCacheControlStringsInvalidDirective() {
    $urls_headers = [
      '/valid-path|invalid-directive',
    ];
    $this->pathValidator->expects($this->any())->method('isValid')->willReturn(true);
    $result = $this->utility->validateCacheControlStrings($urls_headers);
    $this->assertNotEmpty($result, 'Errors should be returned for invalid cache-control directive.');
    $this->assertStringContainsString('This invalid-directive cache-control directive is invalid.', $result[0]);
  }

  /**
   * Tests validateCacheControlStrings with invalid format.
   */
  public function testValidateCacheControlStringsInvalidFormat() {
    $urls_headers = [
      '/valid-path public, max-age=3600',
    ];
    $this->pathValidator->method('isValid')->willReturn(false);
    $result = $this->utility->validateCacheControlStrings($urls_headers);
    $this->assertNotEmpty($result, 'Errors should be returned for invalid format.');
    $this->assertStringContainsString('This /valid-path public, max-age=3600 format is invalid.', $result[0]);
  }

  /**
   * Tests validateCacheControlStrings with invalid internal URL.
   */
  public function testValidateCacheControlStringsInvalidInternalUrl() {
    $urls_headers = [
      '/invalid-internal-path|public, max-age=3600',
    ];
    $this->pathAliasManager->method('getPathByAlias')->willReturn('/invalid-internal-path');
    $this->pathValidator->method('isValid')->willReturn(false);
    $result = $this->utility->validateCacheControlStrings($urls_headers);
    $this->assertNotEmpty($result, 'Errors should be returned for invalid internal URL.');
    $this->assertStringContainsString('This /invalid-internal-path is not a valid internal URL.', $result[0]);
  }

  /**
   * Tests the setTempHeadersInState method.
   */
  public function testSetTempHeadersInState() {
    $this->state->expects($this->once())
      ->method('set')
      ->with('override_cache_control_headers.urls_headers', $this->anything());

    $headers = ['/example|no-cache|10'];
    $this->state->expects($this->once())->method('get')->willReturn([]);
    $this->utility->setTempHeadersInState($headers);
  }

  /**
   * Tests the processTempUrlHeaders method.
   */
  public function testProcessTempUrlHeaders() {
    $urls_header = [
      '/example|no-cache|10|1620000000|1620003600',
      '/expired|no-cache|10|1620000000|1620000001',
    ];

    $state_changed = false;
    $result = $this->utility->processTempUrlHeaders($urls_header, $state_changed);
    $this->assertTrue($state_changed, 'State should be marked as changed.');
  }

  /**
   * Tests the invalidateCache method.
   */
  public function testInvalidateCache() {
    $this->configFactory->expects($this->once())
      ->method('reset')
      ->with('override_cache_control_headers.settings');

    $this->cacheTagsInvalidator->expects($this->once())
      ->method('invalidateTags')
      ->with(['config:override_cache_control_headers.settings']);

    $this->utility->invalidateCache();
  }
}
