<?php

namespace Drupal\page_cache_single\StackMiddleware;

use Drupal\Core\Database\Database;
use Drupal\page_cache\StackMiddleware\PageCache;
use Symfony\Component\HttpFoundation\Request;

/**
 * Extends the PageCache middleware to handle 404 pages differently.
 *
 * This class customizes the caching behavior for 404 pages in Drupal.
 * It ensures that 404 pages are cached separately from other pages
 * by generating unique cache IDs for them. This is useful for improving
 * performance and minimizing size of the cache_page database table.
 */
class PageCacheSingle extends PageCache {

  /**
   * The language prefix detected for the last inspected path, if any.
   *
   * @var string|null
   */
  protected $languagePrefix = NULL;

  /**
   * {@inheritdoc}
   */
  protected function getCacheId(Request $request) {
    if (isset($this->cid)) {
      return $this->cid;
    }

    $path = $request->getPathInfo();
    $languagePrefix = $this->getLanguagePrefixFromPath($path);

    if ($this->isPath404($path)) {
      $cid_parts = [
        $request->getSchemeAndHttpHost(),
        '404',
        $languagePrefix,
      ];
    }
    else {
      $cid_parts = [
        $request->getSchemeAndHttpHost() . $path,
        $request->getRequestFormat(NULL),
        $languagePrefix,
      ];
    }

    $this->cid = implode(':', $cid_parts);
    return $this->cid;
  }

  /**
   * Determines if the given path corresponds to a 404 page.
   *
   * @param string $path
   *   The path to check.
   *
   * @return bool
   *   TRUE if the path is a 404, FALSE otherwise.
   */
  protected function isPath404(string $path) {
    if ($path === '/') {
      return FALSE;
    }

    if (!empty($this->languagePrefix) && strpos($path, '/' . $this->languagePrefix) === 0) {
      $path = substr($path, strlen('/' . $this->languagePrefix));
      if ($path === '') {
        $path = '/';
      }
    }

    if ($path === '/') {
      return FALSE;
    }

    $dbConnection = Database::getConnection();

    $sql = "SELECT pattern_outline FROM {router} WHERE :path LIKE CONCAT(pattern_outline, '%') AND pattern_outline != '/'";
    $result = $dbConnection->query($sql, [':path' => $path])->fetchField();
    if ($result) {
      return FALSE;
    }

    $path_noslash = rtrim($path, '/');
    $sql = "SELECT id FROM {path_alias} WHERE alias = :alias";
    $result = $dbConnection->query($sql, [':alias' => urldecode($path_noslash)])->fetchField();
    if ($result) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Extracts the language code from the URL path.
   *
   * @param string $path
   *   The URL path to analyze.
   *
   * @return string
   *   The language code if found, or 'en' as default.
   */
  protected function getLanguagePrefixFromPath(string $path) {
    if ($path === '/') {
      $this->languagePrefix = 'en';
      return 'en';
    }

    $this->languagePrefix = NULL;

    $dbConnection = Database::getConnection();
    $sql = "SELECT data FROM {config} WHERE name = 'language.negotiation'";
    $result = $dbConnection->query($sql)->fetchField();

    if ($result) {
      $config = unserialize($result, ['allowed_classes' => FALSE]);
      if (isset($config['url']['prefixes'])) {
        $prefixes = $config['url']['prefixes'];
        foreach ($prefixes as $prefix) {
          if (!empty($prefix) && strpos($path, '/' . $prefix) === 0) {
            $this->languagePrefix = $prefix;
            return $prefix;
          }
        }
      }
    }

    // Default to 'en' if no language prefix found.
    $this->languagePrefix = 'en';
    return 'en';
  }

}
