<?php

namespace Drupal\Tests\page_cache_single\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\system\Functional\Cache\AssertPageCacheContextsAndTagsTrait;

/**
 * Functional tests for the PageCacheSingle module.
 *
 * This test ensures that 404 pages and non-404 pages are cached correctly
 * according to the custom caching logic implemented in the module.
 *
 * @group page_cache_single
 */
class PageCacheSingleFunctionalTest extends BrowserTestBase {

  use AssertPageCacheContextsAndTagsTrait;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->createContentType(['type' => 'page']);
  }

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'page_cache',
    'page_cache_single',
    'node',
    'path',
  ];


  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests caching behavior for non existing pages.
   */
  public function testCacheFor404Pages() {
    $this->enablePageCaching();

    $path = '/non-existent-path';

    $this->drupalGet($path);
    $this->assertSession()->responseHeaderEquals('X-Drupal-Cache', 'MISS');
    $this->assertSession()->statusCodeEquals(404);

    $this->drupalGet($path);
    $this->assertSession()->responseHeaderEquals('X-Drupal-Cache', 'HIT');

    $cache = \Drupal::database()->select('cache_page', 'c')
      ->fields('c', ['cid'])
      ->condition('cid', '%404%', 'LIKE')
      ->execute()
      ->fetchField();
    $this->assertNotEmpty($cache);

    $this->drupalGet($path, ['query' => ['foo' => 'bar']]);
    $this->assertSession()->responseHeaderEquals('X-Drupal-Cache', 'HIT');
  }

  /**
   * Tests caching behavior for existing pages.
   */
  public function testCacheForNon404Pages() {
    $this->enablePageCaching();

    $node = $this->drupalCreateNode([
      'title' => 'Test Page',
      'type' => 'page',
      'status' => 1,
    ]);
    $nodePath = 'node/' . $node->id();

    $this->drupalGet($nodePath);
    $this->assertSession()->responseHeaderEquals('X-Drupal-Cache', 'MISS');
    $this->assertSession()->statusCodeEquals(200);

    $this->drupalGet($nodePath);
    $this->assertSession()->responseHeaderEquals('X-Drupal-Cache', 'HIT');

    $cache = \Drupal::database()->select('cache_page', 'c')
      ->fields('c', ['cid'])
      ->condition('cid', '%' . $nodePath . '%', 'LIKE')
      ->execute()
      ->fetchField();
    $this->assertNotEmpty($cache);

    $this->drupalGet($nodePath, ['query' => ['foo' => 'bar']]);
    $this->assertSession()->responseHeaderEquals('X-Drupal-Cache', 'HIT');
  }

}
