<?php

namespace Drupal\page_proxy\Controller;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Cookie\CookieJar;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\RequestOptions;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;

/**
 * Serve page proxy requests.
 */
class PageProxy implements ContainerInjectionInterface {

  /**
   * Constructor.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(
    protected RequestStack $requestStack,
    protected ClientInterface $httpClient,
    protected LoggerChannelInterface $logger,
    protected ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack'),
      $container->get('http_client'),
      $container->get('logger.factory')->get('page_proxy'),
      $container->get('module_handler')
    );
  }

  /**
   * Serve a page as a proxy.
   *
   * @param string $uri
   *   The URI of the page, that should be served.
   * @param string $path
   *   The local base path of the proxied page.
   * @param string $cookie_prefix
   *   Prefix used for proxied cookies.
   * @param array $cookie_allowlist
   *   An array of case-insensitive cookie names that are carried between user
   *   and proxied page.
   * @param array $headers_request
   *   An array of lowercase header names that are transferred from the client's
   *   request to the request to the proxied server.
   * @param array $headers_response
   *   An array of lowercase header names that are transferred from the proxied
   *   server's response to the response to the client.
   * @param string ...
   *   Any subpath of the provided URI can be requested by passing additional
   *   path components as arguments. Each subsequent argument is used as
   *   consequtive path component.
   *   Example: If the URI is https://example.com and the additional arguments
   *   "foo" and "bar" are passed, then https://example.com/foo/bar is served.
   */
  public function serve($uri, $path, $cookie_prefix, $cookie_allowlist, $headers_request, $headers_response) {
    // Ensure that $path meets our assumptions.
    if (substr($path, 0, 1) !== '/') {
      $path = '/' . $path;
    }
    if (substr($path, -1) === '/') {
      $path = substr($path, 0, -1);
    }

    $request = $this->requestStack->getCurrentRequest();
    $request_uri = $request->getRequestUri();
    $request_uri_parts = UrlHelper::parse($request_uri);
    $request_path = $request_uri_parts['path'] ?? '';
    $request_query = UrlHelper::buildQuery($request_uri_parts['query']);
    // If the request is made to the base path and the requested path does not
    // terminate with a slash, redirect to the path ending with a slash. This is
    // necessary for relative URLs on the proxied page to work. A browser will
    // honour the page's path in relative URLs only if it ends with a slash. Ie.
    // if there is an image on the page with href="image.png", the request for
    // the image is made to /image.png, if the page's path is /page. Otherwise,
    // if the page's path is /page/, the request for the image is directed to
    // /page/image.png.
    if ($request_path === $path) {
      $redirect_uri = $path . '/';
      if (!empty($request_uri_parts['query'])) {
        $redirect_uri .= '?' . $request_query;
      }
      if (!empty($request_uri_parts['fragment'])) {
        $redirect_uri .= '#' . $request_uri_parts['fragment'];
      }
      return new RedirectResponse($redirect_uri, 301);
    }

    $subpath = substr($request_path, strlen($path));
    $p_uri_parts = UrlHelper::parse($uri);
    if ($subpath !== '/') {
      if (substr($p_uri_parts['path'], -1) === '/') {
        $p_uri_parts['path'] .= substr($subpath, 1);
      }
      else {
        $p_uri_parts['path'] .= $subpath;
      }
    }
    if (!empty($request_uri_parts['query'])) {
      $p_uri_parts += $request_uri_parts['query'];
    }

    // Get the HTTP headers from the $_SERVER variable. Headers are those keys
    // prefixed with "HTTP_". There is a PHP function getallheaders(), but it is
    // specific for the Apache PHP module and thus does not work in all
    // contexts.
    $headers_clt_in = [];
    foreach ($_SERVER as $k => $v) {
      $k = strtolower($k);
      $k = str_replace('_', '-', $k);
      if (substr($k, 0, 5) === 'http-') {
        $headers_clt_in[substr($k, 5)] = $v;
      }
    }

    // Generate the headers used for the request to the external server. We do
    // not want to simply copy all headers of the request sent to this server,
    // because that might leak sensitive information.
    $headers_srv_out = [];
    foreach ($headers_request as $h) {
      if (!empty($headers_clt_in[$h])) {
        $headers_srv_out[$h] = $headers_clt_in[$h];
      }
    }
    $p_options = [
      RequestOptions::HTTP_ERRORS => FALSE,
      'headers' => $headers_srv_out,
      'body' => NULL,
    ];
    $p_options['method'] = $request->getMethod();
    if ($p_options['method'] === 'POST') {
      $p_options['body'] = $request->getContent();
      if (in_array("content-type", $headers_request)) {
        $p_options['headers']['content-type'] = $request->getContentTypeFormat();
      }
      else {
        $p_options['headers']['content-type'] = 'application/x-www-form-urlencoded';
      }
    }

    // Set cookies from the user's request on the request to the remote server.
    // Only use allowed cookies.
    $cookie_allowlist_in = [];
    foreach ($cookie_allowlist as $c) {
      $cookie_allowlist_in[$cookie_prefix . $c] = $c;
    }
    if ($request->cookies->count() > 0) {
      $cookies_srv_out = [];
      foreach ($request->cookies as $c_name => $c_value) {
        $c_name = strtolower($c_name);
        if (isset($cookie_allowlist_in[$c_name])) {
          $cookies_srv_out[] = $cookie_allowlist_in[$c_name] . '=' . $c_value;
        }
      }
      if (!empty($cookies_srv_out)) {
        $p_options['headers']['cookie'] = implode('; ', $cookies_srv_out);
      }
    }

    $sane_path = strtolower($path);
    if (str_starts_with($sane_path, '/')) {
      $sane_path = substr($sane_path, 1);
    }
    $sane_path = preg_replace('/[^a-zA-Z0-9_]/', '_', $sane_path);
    $p_uri = $p_uri_parts['path'];
    if (!empty($p_uri_parts['query'])) {
      $p_uri .= '?' . UrlHelper::buildQuery($p_uri_parts['query']);
    }
    if (!empty($p_uri_parts['fragment'])) {
      $p_uri .= '#' . $p_uri_parts['fragment'];
    }
    $this->moduleHandler->alter('page_proxy_request', $p_options, $p_uri);
    $this->moduleHandler->alter('page_proxy_request_' . $sane_path, $p_options, $p_uri);
    $p_request = new Request($p_options['method'], $p_uri, $p_options['headers'], $p_options['body']);
    unset($p_options['method']);
    unset($p_options['headers']);
    unset($p_options['body']);
    $p_response = NULL;
    try {
      $p_response = $this->httpClient->send($p_request, $p_options);
    }
    catch (\Exception $e) {
      $this->getLogger('page_proxy')->error('Proxy request failed with exception (@t): @e',
        ['@t' => get_class($e), '@e' => $e->getMessage()]);
    }

    if (empty($p_response)) {
      return (new Response())->setStatusCode(404, 'Not found');
    }

    // Carry some of the headers sent by the external server's response over to
    // the headers of our response to the client.
    $p_response_headers = [];
    foreach ($p_response->getHeaders() as $h => $v) {
      $p_response_headers[strtolower($h)] = $v;
    }
    $response_headers = [];
    foreach ($headers_response as $h) {
      if (isset($p_response_headers[$h])) {
        $response_headers[$h] = $p_response_headers[$h];
      }
    }

    // Carry cookies of the external server's response to the user's response.
    $domain = $request->getHttpHost();
    if (($s = strpos($domain, ':'))) {
      $domain = substr($domain, 0, $s);
    }
    $cookies_srv_in = new CookieJar();
    $cookies_srv_in->extractCookies($p_request, $p_response);
    $response_headers['set-cookie'] = [];
    foreach ($cookies_srv_in as $c) {
      if (!in_array(strtolower($c->getName()), $cookie_allowlist)) {
        continue;
      }
      if (!empty($cookie_prefix)) {
        $c->setName($cookie_prefix . $c->getName());
      }
      $c->setDomain($domain);
      if (empty($c->getPath()) || $c->getPath() === '/') {
        $c->setPath($path);
      }
      elseif (substr($c->getPath(), 0, 1) === '/') {
        $srv_path = parse_url($uri, PHP_URL_PATH);
        if (substr($srv_path, -1) === '/') {
          $srv_path = substr($srv_path, 0, -1);
        }
        $srv_path_len = strlen($srv_path);
        if ($c->getPath() === $srv_path
          || substr($c->getPath(), 0, $srv_path_len + 1) === $srv_path . '/') {

          $c->setPath(substr($c->getPath(), $srv_path_len));
        }
        $c->setPath($path . $c->getPath());
      }
      $response_headers['set-cookie'][] = (string) $c;
    }

    $data = (string) $p_response->getBody();
    // Before printing the body, replace absolute paths in src and href
    // attributes in the HTML code.
    if (!empty($p_response_headers['content-type'][0])) {
      if (substr($p_response_headers['content-type'][0], 0, 9) === 'text/html') {
        $data = preg_replace('#(action|src|href)="/(?!/)#', '$1="/' . $path . '/', $data);
      }
      elseif (substr($p_response_headers['content-type'][0], 0, 8) === 'text/css') {
        $data = preg_replace('#url\("/(?!/)#', 'url("/' . $path . '/', $data);
      }
    }
    $response_info = [
      'status_code' => $p_response->getStatusCode(),
      'reason_phrase' => $p_response->getReasonPhrase(),
      'headers' => $response_headers,
      'body' => $data,
    ];
    $this->moduleHandler->alter('page_proxy_response_' . $sane_path, $response_info, $p_uri);
    $this->moduleHandler->alter('page_proxy_response', $response_info, $p_uri);
    return (new Response($response_info['body'], 200, $response_headers))
      ->setStatusCode($response_info['status_code'], $response_info['reason_phrase'])
      ->prepare($request);
  }

}
