<?php

namespace Drupal\page_proxy\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteBuilderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provide module settings.
 */
class Settings extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'page_proxy_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['page_proxy.settings'];
  }

  /**
   * Constructs a \Drupal\page_proxy\Form\Settings object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Core\Routing\RouteBuilderInterface $routeBuilder
   *   The route builder.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    protected RouteBuilderInterface $routeBuilder,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('router.builder')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('page_proxy.settings');
    $form['page_proxies'] = [
      '#type' => 'fieldset',
      '#tree' => TRUE,
      '#title' => $this->t('Page proxies'),
      '#config_target' => 'page_proxy.settings:page_proxies',
    ];
    $page_proxies = $form_state->getValue('page_proxies')
      ?? ($config->get('page_proxies') ?? []);
    $triggering_element = $form_state->getTriggeringElement();
    foreach ($page_proxies as $k => $p) {
      if (!empty($triggering_element)
        && $triggering_element['#name'] === 'remove_' . $k) {

        continue;
      }
      $form['page_proxies'][$k] = $this->getProxyPageSettings($p, $k);
    }
    if (empty($triggering_element)
      || $triggering_element['#name'] === 'add_more') {

      $form['page_proxies'][] = $this->getProxyPageSettings();
    }
    $form['add_more'] = [
      '#type' => 'button',
      '#value' => $this->t('Add more'),
      '#name' => 'add_more',
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * Get the settings form element for a page proxy.
   *
   * @param array $default_values
   *   (Optional) An array of default values for the element. The values must be
   *   keyed by the element names for which they serve as default.
   * @param mixed $remove_index
   *   (Optional) If provided, a remove button is added to the returned element.
   *
   * @return array
   *   A form fieldset element for input of a page proxy configuration.
   */
  protected function getProxyPageSettings(
    array $default_values = [],
    $remove_index = NULL,
  ) {

    $element = [
      '#type' => 'fieldset',
      'title' => [
        '#type' => 'textfield',
        '#title' => $this->t('Title'),
        '#description' => $this->t("The proxied page's title."),
        '#default_value' => $default_values['title'] ?? '',
      ],
      'path' => [
        '#type' => 'textfield',
        '#title' => $this->t('Path'),
        '#description' => $this->t("The proxied page's path within Drupal."),
        '#default_value' => $default_values['path'] ?? '',
      ],
      'uri' => [
        '#type' => 'textfield',
        '#title' => $this->t('URI'),
        '#description' => $this->t('The URI of the proxied web page.'),
        '#default_value' => $default_values['uri'] ?? '',
      ],
      'max_depth' => [
        '#type' => 'textfield',
        '#title' => $this->t('Maximum link depth'),
        '#description' => $this->t('The maximum depth of paths on the proxied page that are served.'),
        '#default_value' => $default_values['max_depth'] ?? '',
      ],
      'permission' => [
        '#type' => 'textfield',
        '#title' => $this->t('Permission'),
        '#description' => $this->t('The machine name of a permission, that is checked, when the page (or any subpage) is accessed. If left empty, there is no access check for the page (or any subpage)'),
        '#default_value' => $default_values['permission'] ?? '',
      ],
      'cookie_prefix' => [
        '#type' => 'textfield',
        '#title' => $this->t("Cookies' prefix"),
        '#description' => $this->t('All cookie of the proxied site are prefixed with this value. This avoids name clashing of cookies of the Drupal page and the remote site.'),
        '#default_value' => $default_values['cookie_prefix'] ?? '',
      ],
      'cookies' => [
        '#type' => 'textarea',
        '#title' => $this->t('Cookies'),
        '#description' => $this->t('A comma separated list of case insensitive cookie names. The cookies with these names are carried over from the user request to the proxy request and from the proxy response to the user response.'),
        '#default_value' => implode(",\n", $default_values['cookies'] ?? []),
      ],
      'headers_request' => [
        '#type' => 'textarea',
        '#title' => $this->t('Request headers'),
        '#description' => $this->t("A comma separated, case insensistive list of header names. These headers are transferred from the user's request to the request to the remote server."),
        '#default_value' => implode(",\n", $default_values['headers_request'] ?? [
          'user-agent',
          'accept',
          'accept-language',
          'dnt',
        ]),
      ],
      'headers_response' => [
        '#type' => 'textarea',
        '#title' => $this->t('Response headers'),
        '#description' => $this->t("A comma separated, case insensistive list of header names. These headers are transferred from the remote server's response to the reponse to the user."),
        '#default_value' => implode(",\n", $default_values['headers_response'] ?? [
          'content-type',
          'content-length',
          'content-encoding',
          'content-language',
          'cache-control',
          'link',
          'expires',
          'pragma',
          'vary',
        ]),
      ],
    ];
    if ($remove_index !== NULL) {
      $element['remove_' . $remove_index] = [
        '#type' => 'button',
        '#value' => $this->t('Remove'),
        '#name' => 'remove_' . $remove_index,
      ];
    }
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $page_proxies = $form_state->getValue('page_proxies') ?? [];
    foreach ($page_proxies as $k => &$p) {
      if (!empty($p['cookies'])) {
        $p['cookies'] = explode(',', $p['cookies']);
        foreach ($p['cookies'] as &$c) {
          $c = trim(strtolower($c));
        }
      }
      else {
        $p['cookies'] = [];
      }

      if (!empty($p['headers_request'])) {
        $p['headers_request'] = explode(',', $p['headers_request']);
        foreach ($p['headers_request'] as &$c) {
          $c = trim(strtolower($c));
        }
      }
      else {
        $p['headers_request'] = [];
      }

      if (!empty($p['headers_response'])) {
        $p['headers_response'] = explode(',', $p['headers_response']);
        foreach ($p['headers_response'] as &$c) {
          $c = trim(strtolower($c));
        }
      }
      else {
        $p['headers_response'] = [];
      }

      unset($p['remove_' . $k]);

      if (empty($p['title']) && empty($p['path']) && empty($p['uri'])) {
        continue;
      }

      if (empty($p['title'])) {
        $form_state->setErrorByName('page_proxies][' . $k . '][title', $this->t('The title must not be empty.'));
      }

      if (empty($p['path'])) {
        $form_state->setErrorByName('page_proxies][' . $k . '][path', $this->t('The path must not be empty.'));
      }
      elseif (parse_url($p['path'], PHP_URL_PATH) !== $p['path']) {
        $form_state->setErrorByName('page_proxies][' . $k . '][path', $this->t('Invalid path.'));
      }
      elseif (substr($p['path'], 0, 1) !== '/') {
        $form_state->setErrorByName('page_proxies][' . $k . '][path', $this->t('The path must start with a "/".'));
      }
      elseif (substr($p['path'], -1) === '/') {
        $form_state->setErrorByName('page_proxies][' . $k . '][path', $this->t('The path must not end with a "/".'));
      }

      if (empty($p['uri'])) {
        $form_state->setErrorByName('page_proxies][' . $k . '][uri', $this->t('The URI must not be empty.'));
      }
      elseif (!filter_var($p['uri'], FILTER_VALIDATE_URL)) {
        $form_state->setErrorByName('page_proxies][' . $k . '][uri', $this->t('Invalid URI. The URI needs to contain scheme and host.'));
      }

      if (strlen($p['max_depth']) === 0) {
        $p['max_depth'] = '0';
      }
      if (((string) $p['max_depth']) != ((int) $p['max_depth'])
        || $p['max_depth'] < 0) {

        $form_state->setErrorByName('page_proxies][' . $k . '][max_depth',
          $this->t('The maximum depth must be a positive integer.'));
      }
    }
    $form_state->setValue('page_proxies', $page_proxies);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $page_proxies = $form_state->getValue('page_proxies') ?? [];
    $page_proxies = array_filter($page_proxies, function ($p) {
      return !(empty($p['title']) && empty($p['path']) && empty($p['uri']));
    });
    $form_state->setValue('page_proxies', $page_proxies);
    parent::submitForm($form, $form_state);
    $this->routeBuilder->setRebuildNeeded();
  }

}
